/*
 * Decompiled with CFR 0.152.
 */
package org.herac.tuxguitar.io.midi;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.herac.tuxguitar.gm.GMChannelRoute;
import org.herac.tuxguitar.gm.GMChannelRouter;
import org.herac.tuxguitar.io.base.TGFileFormatException;
import org.herac.tuxguitar.io.base.TGSongReader;
import org.herac.tuxguitar.io.base.TGSongReaderHandle;
import org.herac.tuxguitar.io.midi.MidiFileFormat;
import org.herac.tuxguitar.io.midi.MidiFileReader;
import org.herac.tuxguitar.io.midi.MidiSettings;
import org.herac.tuxguitar.io.midi.base.MidiEvent;
import org.herac.tuxguitar.io.midi.base.MidiMessage;
import org.herac.tuxguitar.io.midi.base.MidiSequence;
import org.herac.tuxguitar.io.midi.base.MidiTrack;
import org.herac.tuxguitar.song.factory.TGFactory;
import org.herac.tuxguitar.song.helpers.tuning.TuningManager;
import org.herac.tuxguitar.song.managers.TGSongManager;
import org.herac.tuxguitar.song.models.TGBeat;
import org.herac.tuxguitar.song.models.TGChannel;
import org.herac.tuxguitar.song.models.TGChannelParameter;
import org.herac.tuxguitar.song.models.TGColor;
import org.herac.tuxguitar.song.models.TGDuration;
import org.herac.tuxguitar.song.models.TGMeasure;
import org.herac.tuxguitar.song.models.TGMeasureHeader;
import org.herac.tuxguitar.song.models.TGNote;
import org.herac.tuxguitar.song.models.TGSong;
import org.herac.tuxguitar.song.models.TGString;
import org.herac.tuxguitar.song.models.TGTempo;
import org.herac.tuxguitar.song.models.TGTimeSignature;
import org.herac.tuxguitar.song.models.TGTrack;
import org.herac.tuxguitar.song.models.TGTuning;
import org.herac.tuxguitar.util.TGContext;

public class MidiSongReader
extends MidiFileFormat
implements TGSongReader {
    private static final int MIN_DURATION_VALUE = 64;
    private int resolution;
    private List<TGChannel> channels;
    private List<TGMeasureHeader> headers;
    private List<TGTrack> tracks;
    private List<TempNote> tempNotes;
    private List<TempChannel> tempChannels;
    private List<TrackTuningHelper> trackTuningHelpers;
    private GMChannelRouter channelRouter;
    private MidiSettings settings;
    private TGFactory factory;
    private String sequenceName = "";
    private TGSongManager tgSongManager;
    private TGContext context;

    public MidiSongReader(TGContext context) {
        this.context = context;
    }

    public void read(TGSongReaderHandle handle) throws TGFileFormatException {
        this.settings = null;
        try {
            this.settings = (MidiSettings)handle.getContext().getAttribute(MidiSettings.class.getName());
        }
        catch (Throwable throwable) {
            throwable.printStackTrace();
        }
        if (this.settings == null) {
            this.settings = MidiSettings.getDefaults();
        }
        try {
            this.factory = handle.getFactory();
            MidiSequence sequence = new MidiFileReader().getSequence(handle.getInputStream());
            this.initFields(sequence);
            for (int seqTrackNb = 0; seqTrackNb < sequence.countTracks(); ++seqTrackNb) {
                MidiTrack track = sequence.getTrack(seqTrackNb);
                int trackNumber = this.getNextTrackNumber();
                int eventsCount = track.size();
                long previousTick = -1L;
                ArrayList<MidiEvent> tickEvents = new ArrayList<MidiEvent>();
                for (int eventNb = 0; eventNb < eventsCount; ++eventNb) {
                    MidiEvent event = track.get(eventNb);
                    if (event.getTick() > previousTick) {
                        this.parseEvents(tickEvents, seqTrackNb, trackNumber);
                    }
                    tickEvents.add(event);
                    previousTick = event.getTick();
                }
                this.parseEvents(tickEvents, seqTrackNb, trackNumber);
            }
            this.tgSongManager = new TGSongManager(this.factory);
            TGSong tgSong = this.factory.newSong();
            tgSong.setName(this.sequenceName);
            this.checkAll(this.tgSongManager);
            Iterator<TGChannel> channels = this.channels.iterator();
            while (channels.hasNext()) {
                tgSong.addChannel(channels.next());
            }
            Iterator<TGMeasureHeader> headers = this.headers.iterator();
            while (headers.hasNext()) {
                tgSong.addMeasureHeader(headers.next());
            }
            Iterator<TGTrack> tracks = this.tracks.iterator();
            while (tracks.hasNext()) {
                tgSong.addTrack(tracks.next());
            }
            handle.setSong(new SongAdjuster(this.factory, tgSong).adjustSong());
        }
        catch (Throwable throwable) {
            throw new TGFileFormatException(throwable);
        }
    }

    private void parseEvents(List<MidiEvent> events, int seqTrackNb, int trackNumber) {
        ArrayList<MidiEvent> handledEvents = new ArrayList<MidiEvent>();
        for (MidiEvent event : events) {
            if (!this.isNoteOff(event)) continue;
            this.parseMessage(seqTrackNb, trackNumber, event.getTick(), event.getMessage());
            handledEvents.add(event);
        }
        for (MidiEvent event : handledEvents) {
            events.remove(event);
        }
        for (MidiEvent event : events) {
            this.parseMessage(seqTrackNb, trackNumber, event.getTick(), event.getMessage());
        }
        events.clear();
    }

    private boolean isNoteOff(MidiEvent event) {
        MidiMessage message = event.getMessage();
        if (message.getType() == 1 && message.getCommand() == 128) {
            return true;
        }
        if (message.getType() == 1 && message.getCommand() == 144) {
            int length = message.getData().length;
            int velocity = length > 2 ? message.getData()[2] & 0xFF : 0;
            return velocity == 0;
        }
        return false;
    }

    private void initFields(MidiSequence sequence) {
        this.resolution = sequence.getResolution();
        this.channels = new ArrayList<TGChannel>();
        this.headers = new ArrayList<TGMeasureHeader>();
        this.tracks = new ArrayList<TGTrack>();
        this.tempNotes = new ArrayList<TempNote>();
        this.tempChannels = new ArrayList<TempChannel>();
        this.trackTuningHelpers = new ArrayList<TrackTuningHelper>();
        this.channelRouter = new GMChannelRouter();
    }

    private int getNextTrackNumber() {
        return this.tracks.size() + 1;
    }

    private void parseMessage(int trackIdx, int trackNumber, long tick, MidiMessage message) {
        long parsedTick = this.parseTick(tick + (long)this.resolution);
        if (message.getType() == 1 && message.getCommand() == 144) {
            this.parseNoteOn(trackNumber, parsedTick, message.getData());
        } else if (message.getType() == 1 && message.getCommand() == 128) {
            this.parseNoteOff(trackNumber, parsedTick, message.getData());
        } else if (message.getType() == 1 && message.getCommand() == 192) {
            this.parseProgramChange(message.getData());
        } else if (message.getType() == 1 && message.getCommand() == 176) {
            this.parseControlChange(message.getData());
        } else if (message.getType() == 2 && message.getCommand() == 3) {
            this.parseTrackName(trackIdx, trackNumber, message.getData());
        } else if (message.getType() == 2 && message.getCommand() == 88) {
            this.parseTimeSignature(parsedTick, message.getData());
        } else if (message.getType() == 2 && message.getCommand() == 81) {
            this.parseTempo(parsedTick, message.getData());
        }
    }

    private long parseTick(long tick) {
        return Math.abs(960L * tick / (long)this.resolution);
    }

    private void parseNoteOn(int track, long tick, byte[] data) {
        int velocity;
        int length = data.length;
        int channel = length > 0 ? data[0] & 0xFF & 0xF : 0;
        int value = length > 1 ? data[1] & 0xFF : 0;
        int n = velocity = length > 2 ? data[2] & 0xFF : 0;
        if (velocity == 0) {
            this.parseNoteOff(track, tick, data);
        } else if (value > 0) {
            this.createTempNotesBefore(tick, track);
            this.getTempChannel(channel).addTrack(track);
            this.getTrackTuningHelper(track).checkValue(value + this.settings.getTranspose());
            this.tempNotes.add(new TempNote(track, channel, value, tick));
        }
    }

    private void parseNoteOff(int track, long tick, byte[] data) {
        int length = data.length;
        int channel = length > 0 ? data[0] & 0xFF & 0xF : 0;
        int value = length > 1 ? data[1] & 0xFF : 0;
        this.createNote(tick, track, channel, value);
    }

    private void parseProgramChange(byte[] data) {
        int instrument;
        int length = data.length;
        int channel = length > 0 ? data[0] & 0xFF & 0xF : -1;
        int n = instrument = length > 1 ? data[1] & 0xFF : -1;
        if (channel != -1 && instrument != -1) {
            this.getTempChannel(channel).setInstrument(instrument);
        }
    }

    private void parseControlChange(byte[] data) {
        int value;
        int length = data.length;
        int channel = length > 0 ? data[0] & 0xFF & 0xF : -1;
        int control = length > 1 ? data[1] & 0xFF : -1;
        int n = value = length > 2 ? data[2] & 0xFF : -1;
        if (channel != -1 && control != -1 && value != -1) {
            if (control == 7) {
                this.getTempChannel(channel).setVolume(value);
            } else if (control == 10) {
                this.getTempChannel(channel).setBalance(value);
            }
        }
    }

    private void parseTrackName(int trackIdx, int trackNumber, byte[] data) {
        String name = new String(data, Charset.forName("UTF-8"));
        if (trackIdx == 0) {
            this.sequenceName = name;
        } else {
            TGTrack track = this.getTrack(trackNumber);
            track.setName(name);
        }
    }

    private void parseTimeSignature(long tick, byte[] data) {
        if (data.length >= 2) {
            TGTimeSignature timeSignature = this.factory.newTimeSignature();
            timeSignature.setNumerator((int)data[0]);
            timeSignature.getDenominator().setValue(4);
            if (data[1] == 0) {
                timeSignature.getDenominator().setValue(1);
            } else if (data[1] == 1) {
                timeSignature.getDenominator().setValue(2);
            } else if (data[1] == 2) {
                timeSignature.getDenominator().setValue(4);
            } else if (data[1] == 3) {
                timeSignature.getDenominator().setValue(8);
            } else if (data[1] == 4) {
                timeSignature.getDenominator().setValue(16);
            } else if (data[1] == 5) {
                timeSignature.getDenominator().setValue(32);
            }
            this.getHeader(tick).setTimeSignature(timeSignature);
        }
    }

    private void parseTempo(long tick, byte[] data) {
        if (data.length >= 3) {
            TGTempo tempo = TGTempo.fromUSQ((TGFactory)this.factory, (int)(data[2] & 0xFF | (data[1] & 0xFF) << 8 | (data[0] & 0xFF) << 16));
            this.getHeader(tick).setTempo(tempo);
        }
    }

    private TGTrack getTrack(int number) {
        TGTrack track2;
        for (TGTrack track2 : this.tracks) {
            if (track2.getNumber() != number) continue;
            return track2;
        }
        track2 = this.factory.newTrack();
        track2.setNumber(number);
        track2.setChannelId(-1);
        track2.getColor().copyFrom(TGColor.RED);
        this.tracks.add(track2);
        return track2;
    }

    private TGMeasureHeader getHeader(long tick) {
        long realTick = tick >= 960L ? tick : 960L;
        for (TGMeasureHeader header : this.headers) {
            if (realTick < header.getStart() || realTick >= header.getStart() + header.getLength()) continue;
            return header;
        }
        TGMeasureHeader last = this.getLastHeader();
        TGMeasureHeader header = this.factory.newHeader();
        header.setNumber(last != null ? last.getNumber() + 1 : 1);
        header.setStart(last != null ? last.getStart() + last.getLength() : 960L);
        header.getTempo().setValue(last != null ? last.getTempo().getValue() : 120);
        if (last != null) {
            header.getTimeSignature().copyFrom(last.getTimeSignature());
        } else {
            header.getTimeSignature().setNumerator(4);
            header.getTimeSignature().getDenominator().setValue(4);
        }
        this.headers.add(header);
        if (realTick >= header.getStart() && realTick < header.getStart() + header.getLength()) {
            return header;
        }
        return this.getHeader(realTick);
    }

    private TGMeasureHeader getLastHeader() {
        if (!this.headers.isEmpty()) {
            return this.headers.get(this.headers.size() - 1);
        }
        return null;
    }

    private TGMeasure getMeasure(TGTrack track, long tick) {
        long realTick = tick >= 960L ? tick : 960L;
        Iterator it = track.getMeasures();
        while (it.hasNext()) {
            TGMeasure measure = (TGMeasure)it.next();
            if (realTick < measure.getStart() || realTick >= measure.getStart() + measure.getLength()) continue;
            return measure;
        }
        this.getHeader(realTick);
        for (int i = 0; i < this.headers.size(); ++i) {
            boolean exist = false;
            TGMeasureHeader header = this.headers.get(i);
            int measureCount = track.countMeasures();
            for (int j = 0; j < measureCount; ++j) {
                TGMeasure measure = track.getMeasure(j);
                if (!measure.getHeader().equals(header)) continue;
                exist = true;
            }
            if (exist) continue;
            TGMeasure measure = this.factory.newMeasure(header);
            track.addMeasure(measure);
        }
        return this.getMeasure(track, realTick);
    }

    private TGBeat getBeat(TGMeasure measure, long start) {
        int beatCount = measure.countBeats();
        for (int i = 0; i < beatCount; ++i) {
            TGBeat beat = measure.getBeat(i);
            if (beat.getStart() != start) continue;
            return beat;
        }
        TGBeat beat = this.factory.newBeat();
        beat.setStart(start);
        measure.addBeat(beat);
        return beat;
    }

    private TempNote getTempNote(int track, int channel, int value, boolean purge) {
        for (int i = 0; i < this.tempNotes.size(); ++i) {
            TempNote note = this.tempNotes.get(i);
            if (note.getTrack() != track || note.getChannel() != channel || note.getValue() != value) continue;
            if (purge) {
                this.tempNotes.remove(i);
            }
            return note;
        }
        return null;
    }

    protected TrackTuningHelper getTrackTuningHelper(int track) {
        TrackTuningHelper helper2;
        for (TrackTuningHelper helper2 : this.trackTuningHelpers) {
            if (helper2.getTrack() != track) continue;
            return helper2;
        }
        helper2 = new TrackTuningHelper(track);
        this.trackTuningHelpers.add(helper2);
        return helper2;
    }

    private void createTempNotesBefore(long tick, int track) {
        long nextTick = tick;
        boolean check = true;
        block0: while (check) {
            check = false;
            for (int i = 0; i < this.tempNotes.size(); ++i) {
                TempNote note = this.tempNotes.get(i);
                if (note.getTick() >= nextTick || note.getTrack() != track) continue;
                nextTick = note.getTick() + 4800L;
                this.createNote(nextTick, track, note.getChannel(), note.getValue());
                check = true;
                continue block0;
            }
        }
    }

    private void createNote(long tick, int track, int channel, int value) {
        TempNote tempNote = this.getTempNote(track, channel, value, true);
        if (tempNote != null) {
            int nValue = tempNote.getValue() + this.settings.getTranspose();
            int nVelocity = 64;
            long nStart = tempNote.getTick();
            while (nStart < tick) {
                TGMeasure measure = this.getMeasure(this.getTrack(track), nStart);
                TGDuration minDuration = this.newDuration(64);
                TGDuration nDuration = TGDuration.fromTime((TGFactory)this.factory, (long)Math.min(tick - nStart, measure.getLength()), (TGDuration)minDuration);
                TGBeat beat = this.getBeat(measure, nStart);
                beat.getVoice(0).getDuration().copyFrom(nDuration);
                TGNote note = this.factory.newNote();
                note.setValue(nValue);
                note.setString(1);
                note.setVelocity(nVelocity);
                note.setTiedNote(nStart > tempNote.getTick());
                beat.getVoice(0).addNote(note);
                nStart = measure.getStart() + measure.getLength();
            }
        }
    }

    public TempChannel getTempChannel(int channel) {
        TempChannel tempChannel2;
        for (TempChannel tempChannel2 : this.tempChannels) {
            if (tempChannel2.getChannel() != channel) continue;
            return tempChannel2;
        }
        tempChannel2 = new TempChannel(channel);
        this.tempChannels.add(tempChannel2);
        return tempChannel2;
    }

    private void checkAll(TGSongManager songManager) throws Exception {
        this.checkChannels();
        this.checkTracks(songManager);
        int headerCount = this.headers.size();
        for (int i = 0; i < this.tracks.size(); ++i) {
            TGTrack track = this.tracks.get(i);
            while (track.countMeasures() < headerCount) {
                TGMeasure lastMeasure;
                long start = 960L;
                TGMeasure tGMeasure = lastMeasure = track.countMeasures() > 0 ? track.getMeasure(track.countMeasures() - 1) : null;
                if (lastMeasure != null) {
                    start = lastMeasure.getStart() + lastMeasure.getLength();
                }
                track.addMeasure(this.factory.newMeasure(this.getHeader(start)));
            }
        }
        if (this.headers.isEmpty() || this.tracks.isEmpty()) {
            throw new Exception("Empty Song");
        }
    }

    private void checkChannels() {
        for (int tc = 0; tc < this.tempChannels.size(); ++tc) {
            TempChannel tempChannel = this.tempChannels.get(tc);
            if (tempChannel.getTracks().isEmpty()) continue;
            boolean channelExists = false;
            for (int c = 0; c < this.channels.size(); ++c) {
                TGChannel tgChannel = this.channels.get(c);
                GMChannelRoute gmChannelRoute = this.channelRouter.getRoute(tgChannel.getChannelId());
                if (gmChannelRoute == null || gmChannelRoute.getChannel1() != tempChannel.getChannel() && gmChannelRoute.getChannel2() != tempChannel.getChannel()) continue;
                channelExists = true;
            }
            if (channelExists) continue;
            TGChannel tgChannel = this.factory.newChannel();
            tgChannel.setChannelId(this.channels.size() + 1);
            tgChannel.setProgram((short)tempChannel.getInstrument());
            tgChannel.setVolume((short)tempChannel.getVolume());
            tgChannel.setBalance((short)tempChannel.getBalance());
            tgChannel.setName("#" + tgChannel.getChannelId());
            tgChannel.setBank(tempChannel.getChannel() == 9 ? (short)128 : 0);
            GMChannelRoute gmChannelRoute = new GMChannelRoute(tgChannel.getChannelId());
            gmChannelRoute.setChannel1(tempChannel.getChannel());
            gmChannelRoute.setChannel2(tempChannel.getChannel());
            for (int tcAux = tc + 1; tcAux < this.tempChannels.size(); ++tcAux) {
                TempChannel tempChannelAux = this.tempChannels.get(tcAux);
                if (!tempChannel.getTracks().equals(tempChannelAux.getTracks())) continue;
                if (gmChannelRoute.getChannel2() == gmChannelRoute.getChannel1()) {
                    gmChannelRoute.setChannel2(tempChannelAux.getChannel());
                    continue;
                }
                tempChannelAux.clearTracks();
            }
            this.channelRouter.configureRoutes(gmChannelRoute, tempChannel.getChannel() == 9);
            TGChannelParameter gmChannel1Param = this.factory.newChannelParameter();
            TGChannelParameter gmChannel2Param = this.factory.newChannelParameter();
            gmChannel1Param.setKey("gm-channel-1");
            gmChannel1Param.setValue(Integer.toString(gmChannelRoute.getChannel1()));
            gmChannel2Param.setKey("gm-channel-2");
            gmChannel2Param.setValue(Integer.toString(gmChannelRoute.getChannel2()));
            tgChannel.addParameter(gmChannel1Param);
            tgChannel.addParameter(gmChannel2Param);
            this.channels.add(tgChannel);
        }
    }

    private void checkTracks(TGSongManager songManager) {
        for (TGTrack track : this.tracks) {
            TGChannel trackChannel = null;
            for (TempChannel tempChannel : this.tempChannels) {
                if (!tempChannel.getTracks().contains(track.getNumber())) continue;
                for (TGChannel tgChannel : this.channels) {
                    GMChannelRoute gmChannelRoute = this.channelRouter.getRoute(tgChannel.getChannelId());
                    if (gmChannelRoute == null || tempChannel.getChannel() != gmChannelRoute.getChannel1()) continue;
                    trackChannel = tgChannel;
                }
            }
            if (trackChannel != null) {
                track.setChannelId(trackChannel.getChannelId());
            }
            if (trackChannel != null && trackChannel.isPercussionChannel()) {
                track.setStrings(songManager.createPercussionStrings(6));
                continue;
            }
            track.setStrings(this.getTrackTuningHelper(track.getNumber()).getStrings(track.getMaxFret()));
        }
    }

    protected TGDuration newDuration(int value) {
        TGDuration duration = this.factory.newDuration();
        duration.setValue(value);
        return duration;
    }

    private class SongAdjuster {
        private TGFactory factory;
        private TGSong song;
        private long minDurationTime;

        public SongAdjuster(TGFactory factory, TGSong song) {
            this.factory = factory;
            this.song = song;
            this.minDurationTime = 40L;
        }

        public TGSong adjustSong() {
            Iterator it = this.song.getTracks();
            while (it.hasNext()) {
                TGTrack track = (TGTrack)it.next();
                this.adjustTrack(track);
            }
            return this.song;
        }

        private void adjustTrack(TGTrack track) {
            Iterator it = track.getMeasures();
            while (it.hasNext()) {
                TGMeasure measure = (TGMeasure)it.next();
                this.process(measure, track.isPercussion(), track.getMaxFret());
            }
        }

        private void process(TGMeasure measure, boolean isPercussionTrack, int maxFret) {
            this.orderBeats(measure);
            this.joinBeats(measure);
            this.adjustStrings(measure, isPercussionTrack, maxFret);
        }

        private void joinBeats(TGMeasure measure) {
            TGBeat previous = null;
            boolean finish = true;
            long measureStart = measure.getStart();
            long measureEnd = measureStart + measure.getLength();
            for (int i = 0; i < measure.countBeats(); ++i) {
                TGBeat beat = measure.getBeat(i);
                long beatStart = beat.getStart();
                long beatLength = beat.getVoice(0).getDuration().getTime();
                if (previous != null) {
                    long previousStart = previous.getStart();
                    long previousLength = previous.getVoice(0).getDuration().getTime();
                    if (beatStart >= previousStart && previousStart + this.minDurationTime > beatStart) {
                        for (int n = 0; n < beat.getVoice(0).countNotes(); ++n) {
                            TGNote note = beat.getVoice(0).getNote(n);
                            previous.getVoice(0).addNote(note);
                        }
                        if (!previous.isChordBeat() && beat.isChordBeat()) {
                            previous.setChord(beat.getChord());
                        }
                        if (!previous.isTextBeat() && beat.isTextBeat()) {
                            previous.setText(beat.getText());
                        }
                        if (beatLength > previousLength && beatStart + beatLength <= measureEnd) {
                            previous.getVoice(0).getDuration().copyFrom(beat.getVoice(0).getDuration());
                        }
                        measure.removeBeat(beat);
                        finish = false;
                        break;
                    }
                    if (previousStart < beatStart && previousStart + previousLength > beatStart) {
                        if (beat.getVoice(0).isRestVoice()) {
                            measure.removeBeat(beat);
                            finish = false;
                            break;
                        }
                        TGDuration duration = TGDuration.fromTime((TGFactory)this.factory, (long)(beatStart - previousStart));
                        previous.getVoice(0).getDuration().copyFrom(duration);
                    }
                }
                if (beatStart + beatLength > measureEnd) {
                    if (beat.getVoice(0).isRestVoice()) {
                        measure.removeBeat(beat);
                        finish = false;
                        break;
                    }
                    TGDuration duration = TGDuration.fromTime((TGFactory)this.factory, (long)(measureEnd - beatStart));
                    beat.getVoice(0).getDuration().copyFrom(duration);
                }
                previous = beat;
            }
            if (!finish) {
                this.joinBeats(measure);
            }
        }

        private void orderBeats(TGMeasure measure) {
            for (int i = 0; i < measure.countBeats(); ++i) {
                TGBeat minBeat = null;
                for (int j = i; j < measure.countBeats(); ++j) {
                    TGBeat beat = measure.getBeat(j);
                    if (minBeat != null && beat.getStart() >= minBeat.getStart()) continue;
                    minBeat = beat;
                }
                measure.moveBeat(i, minBeat);
            }
        }

        private void adjustStrings(TGMeasure measure, boolean isPercussionTrack, int maxFret) {
            TGString string = this.factory.newString();
            string.setNumber(1);
            string.setValue(0);
            ArrayList<Integer> strings = new ArrayList<Integer>();
            strings.add(string.getValue());
            if (isPercussionTrack) {
                MidiSongReader.this.tgSongManager.getTrackManager().allocatePercussionNotesToStrings(strings, measure.getBeats(), measure.getTrack().getStrings());
            } else {
                MidiSongReader.this.tgSongManager.getTrackManager().allocateNotesToStrings(strings, measure.getBeats(), measure.getTrack().getStrings(), maxFret);
            }
        }
    }

    private class TempChannel {
        private int channel;
        private int instrument;
        private int volume;
        private int balance;
        private Set<Integer> tracks;

        public TempChannel(int channel) {
            this.channel = channel;
            this.instrument = 0;
            this.volume = 127;
            this.balance = 64;
            this.tracks = new HashSet<Integer>();
        }

        public int getBalance() {
            return this.balance;
        }

        public void setBalance(int balance) {
            this.balance = balance;
        }

        public int getChannel() {
            return this.channel;
        }

        public int getInstrument() {
            return this.instrument;
        }

        public void setInstrument(int instrument) {
            this.instrument = instrument;
        }

        public Set<Integer> getTracks() {
            return this.tracks;
        }

        public void addTrack(int track) {
            this.tracks.add(track);
        }

        public void clearTracks() {
            this.tracks.clear();
        }

        public int getVolume() {
            return this.volume;
        }

        public void setVolume(int volume) {
            this.volume = volume;
        }
    }

    private class TrackTuningHelper {
        private int track;
        private int maxValue;
        private int minValue;

        public TrackTuningHelper(int track) {
            this.track = track;
            this.maxValue = -1;
            this.minValue = -1;
        }

        public void checkValue(int value) {
            if (this.minValue < 0 || value < this.minValue) {
                this.minValue = value;
            }
            if (this.maxValue < 0 || value > this.maxValue) {
                this.maxValue = value;
            }
        }

        public List<TGString> getStrings(int maxFret) {
            ArrayList<TGString> strings = new ArrayList<TGString>();
            TuningManager tuningManager = TuningManager.getInstance((TGContext)MidiSongReader.this.context);
            List tunings = tuningManager.getPriorityTgTunings();
            for (TGTuning tuning : tunings) {
                if (!tuning.isWithinRange(this.minValue, this.maxValue - maxFret)) continue;
                int[] notes = tuning.getValues();
                for (int i = 0; i < notes.length; ++i) {
                    strings.add(TGSongManager.newString((TGFactory)MidiSongReader.this.factory, (int)(i + 1), (int)notes[i]));
                }
            }
            if (strings.isEmpty()) {
                int stringCount = 6;
                int stringSpacing = (this.maxValue - (maxFret - 4) - this.minValue) / stringCount;
                if (stringSpacing > 5) {
                    stringCount = 7;
                    stringSpacing = (this.maxValue - (maxFret - 4) - this.minValue) / stringCount;
                }
                int maxStringValue = this.minValue + stringCount * stringSpacing;
                while (strings.size() < stringCount) {
                    strings.add(TGSongManager.newString((TGFactory)MidiSongReader.this.factory, (int)(strings.size() + 1), (int)(maxStringValue -= stringSpacing)));
                }
            }
            return strings;
        }

        public int getTrack() {
            return this.track;
        }
    }

    private class TempNote {
        private int track;
        private int channel;
        private int value;
        private long tick;

        public TempNote(int track, int channel, int value, long tick) {
            this.track = track;
            this.channel = channel;
            this.value = value;
            this.tick = tick;
        }

        public int getChannel() {
            return this.channel;
        }

        public long getTick() {
            return this.tick;
        }

        public int getTrack() {
            return this.track;
        }

        public int getValue() {
            return this.value;
        }
    }
}

