/****************************************************************************
 *   This file is part of the aGrUM/pyAgrum library.                        *
 *                                                                          *
 *   Copyright (c) 2005-2025 by                                             *
 *       - Pierre-Henri WUILLEMIN(_at_LIP6)                                 *
 *       - Christophe GONZALES(_at_AMU)                                     *
 *                                                                          *
 *   The aGrUM/pyAgrum library is free software; you can redistribute it    *
 *   and/or modify it under the terms of either :                           *
 *                                                                          *
 *    - the GNU Lesser General Public License as published by               *
 *      the Free Software Foundation, either version 3 of the License,      *
 *      or (at your option) any later version,                              *
 *    - the MIT license (MIT),                                              *
 *    - or both in dual license, as here.                                   *
 *                                                                          *
 *   (see https://agrum.gitlab.io/articles/dual-licenses-lgplv3mit.html)    *
 *                                                                          *
 *   This aGrUM/pyAgrum library is distributed in the hope that it will be  *
 *   useful, but WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,          *
 *   INCLUDING BUT NOT LIMITED TO THE WARRANTIES MERCHANTABILITY or FITNESS *
 *   FOR A PARTICULAR PURPOSE  AND NONINFRINGEMENT. IN NO EVENT SHALL THE   *
 *   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 *   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,        *
 *   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR  *
 *   OTHER DEALINGS IN THE SOFTWARE.                                        *
 *                                                                          *
 *   See LICENCES for more details.                                         *
 *                                                                          *
 *   SPDX-FileCopyrightText: Copyright 2005-2025                            *
 *       - Pierre-Henri WUILLEMIN(_at_LIP6)                                 *
 *       - Christophe GONZALES(_at_AMU)                                     *
 *   SPDX-License-Identifier: LGPL-3.0-or-later OR MIT                      *
 *                                                                          *
 *   Contact  : info_at_agrum_dot_org                                       *
 *   homepage : http://agrum.gitlab.io                                      *
 *   gitlab   : https://gitlab.com/agrumery/agrum                           *
 *                                                                          *
 ****************************************************************************/


/**
 * @file
 * @brief the class for computing the log2 of the parametric complexity of
 * an r-ary multinomial variable
 *
 * @author Christophe GONZALES(_at_AMU) and Pierre-Henri WUILLEMIN(_at_LIP6)
 */

#include <agrum/base/core/math/variableLog2ParamComplexity.h>

#ifdef GUM_NO_INLINE
#  include <agrum/base/core/math/variableLog2ParamComplexity_inl.h>
#endif   // GUM_NO_INLINE

namespace gum {

  /// returns the value of the log in base 2 of Cnr
  double VariableLog2ParamComplexity::log2Cnr(const std::size_t r, const double n) {
    // we know that c_n^1 = 1 for all values of n
    // in addition, c_0^r = 1 for all values of r
    // finally, it is easy to see that c_1^r = r for all r
    if (r == std::size_t(1)) return 0.0;         // log2(1)
    if (n == 0.0) return 0.0;                    // log2(1)
    if (n == 1.0) return std::log2((double)r);   // log2(r)

    if (n < 0.0) {
      GUM_ERROR(OutOfBounds,
                "In the penalty of the fNML score, n must be greater "
                    << "than or equal to 0. But, here, n = " << n);
    }

    if (n < VariableLog2ParamComplexityCTableNSize) {
      // check if we can find the value we look for in precomputed table
      // ScorefNMLVariableLog2ParamComplexity
      std::size_t xn = (std::size_t)n;
      if (r - 2 < VariableLog2ParamComplexityCTableRSize) {
        return VariableLog2ParamComplexityCTable[r - 2][xn];
      } else {
        // try to find the value in the cache
        if (_use_cache_) {
          try {
            return _cache_[std::pair< std::size_t, double >{r, n}];
          } catch (NotFound const&) {}
        }

        // use Equation (13) of the paper to compute the value of cnr:
        // C_n^r = C_n^{r-1} + (n / (r-2)) C_n^{r-2}
        // as we handle only log2's of C_n^r, we have the following:
        // let k_r be such that C_n^{r-2} = k_r * C_n^{r-1}
        // log2 ( C_n^r ) = log2 ( C_n^{r-1} + k_r * (n/(r-2)) * C_n^{r-1} )
        //                = log2 ( C_n^{r-1} ) + log2 ( 1 + k_r * (n/(r-2)) )
        // as  k_r = C_n^{r-2} / C_n^{r-1}, we have that
        // log2(k_r) = log2 ( C_n^{r-2} ) - log2 ( C_n^{r-1} )
        // so, k_r = exp ( (log2(cn_^{r-2}) - log2(C_n^{r-1})) * log(2) )
        // now, let q_r = 1 + k_r * (n/(r-2)), then
        // C_n^r = C_n^{r-1} * q_r, or, equivalently,
        // log2(C_n^r) = log2(C_n^{r-1}) + log2(q_r)
        // Now, we can use the same method to compute C_n^{r+1}:
        // k_{r+1}   = C_n^{r-1} / C_n^r = 1 / q_r
        // q_{r+1}   = 1 + k_{r+1} * (n/(r-1))
        // C_n^{r+1} = C_n^r * q_{r+1}
        double log2Cnr1 = VariableLog2ParamComplexityCTable[3][xn];   // log(C_n^5)
        double log2Cnr2 = VariableLog2ParamComplexityCTable[2][xn];   // log(C_n^4)
        double log2Cnr  = 0.0;
        double k_r      = std::exp((log2Cnr2 - log2Cnr1) * M_LN2);
        double q_r      = 1.0 + k_r * n / (6.0 - 2.0);                // we first compute C_n^6
        for (std::size_t i = std::size_t(6); i <= r; ++i) {
          log2Cnr  = log2Cnr1 + std::log2(q_r);
          log2Cnr1 = log2Cnr;
          k_r      = 1.0 / q_r;
          q_r      = 1.0 + k_r * (n / (i - 1.0));
        }

        // if we use a cache, update it
        if (_use_cache_) { _cache_.insert(std::pair< std::size_t, double >{r, n}, log2Cnr); }

        return log2Cnr;
      }
    } else {
      // try to find the value in the cache
      if (_use_cache_) {
        try {
          return _cache_[std::pair< std::size_t, double >{r, n}];
        } catch (NotFound const&) {}
      }

      // compute the corrected Szpankowski approximation of cn2 (see the
      // documentation of constants cst1, cst2, cst3 in the ScorefNML header)
      double log2Cnr1 = 0.5 * std::log2(n) + _cst1_ + _cst2_ / std::sqrt(n) + _cst3_ / n;
      if (r == std::size_t(2)) return log2Cnr1;

      // the value of log2(cn1), which is always equal to 0
      double log2Cnr2 = 0.0;

      // use Equation (13) of the paper to compute the value of cnr
      // (see the detail of the formulas in the above if statement)
      double k_r     = std::exp((log2Cnr2 - log2Cnr1) * M_LN2);
      double q_r     = 1.0 + k_r * n / (3.0 - 2.0);   // we first compute C_n^3
      double log2Cnr = 0.0;
      for (std::size_t i = std::size_t(3); i <= r; ++i) {
        log2Cnr  = log2Cnr1 + std::log2(q_r);
        log2Cnr1 = log2Cnr;
        k_r      = 1.0 / q_r;
        q_r      = 1.0 + k_r * (n / (i - 1.0));
      }

      // if we use a cache, update it
      if (_use_cache_) { _cache_.insert(std::pair< std::size_t, double >{r, n}, log2Cnr); }

      return log2Cnr;
    }
  }

  /// the function used to write the cpp file with the values of log2(Cnr)
  void VariableLog2ParamComplexity::CnrToFile(const std::string& filename) {
    // save all the value of cn2
    std::vector< long double > cn2_table(VariableLog2ParamComplexityCTableNSize);
    cn2_table[0] = 1;
    cn2_table[1] = 2;

    // for every value of n less than Szpankowski_threshold, we compute the
    // value of C_n^2 and write it into cn2_table
    GammaLog2 gamma_log2;
    for (double n = 2; n < VariableLog2ParamComplexityCTableNSize; ++n) {
      // here, note that, in  Silander, Roos, Kontkanen and Myllymaki (2007)
      // "Factorized Normalized Maximum Likelihood Criterion for Learning
      // Bayesian network Structures" paper, there is an uppercase N in the
      // formula, but this should be a lowercase n. In addition, we will loop
      // only on h=1 to n-1 and add to 2.0 the value computed to take into
      // account of h=0 and h=n.
      long double cn2 = 2;
      for (double h = 1; h < n; ++h) {
        long double elt = (gamma_log2(n + 1) - gamma_log2(h + 1) - gamma_log2((n - h) + 1)) * M_LN2
                        + h * std::log(h / n) + (n - h) * std::log((n - h) / n);
        cn2 += std::exp(elt);
      }

      // const double logCn2 = (double) std::log2 ( cn2 );

      cn2_table[int(n)] = cn2;
    }

    // write the header of the output file
    std::ofstream outfile(filename);
    if (!outfile.is_open()) { GUM_ERROR(IOError, "It is impossible to open file " << filename) }
    outfile.precision(20);
    outfile << "namespace gum {\n\n";
    /*
      outfile << "  // the size in r of VariableLog2ParamComplexityCTable\n";
      outfile << "  const std::size_t VariableLog2ParamComplexityCTableRSize = "
      << "4;\n\n";
      outfile << "  // the size in n of VariableLog2ParamComplexityCTable\n";
      outfile << "  const std::size_t VariableLog2ParamComplexityCTableNSize = "
      << VariableLog2ParamComplexityCTableNSize << ";\n\n";
    */
    outfile << "  // the CTable cache for log2(Cnr), n < " << VariableLog2ParamComplexityCTableNSize
            << " and r in {2,3,4,5}\n";
    outfile << "  const double VariableLog2ParamComplexityCTable[4]["
            << VariableLog2ParamComplexityCTableNSize << "] = {\n";

    // write the values of Cn2:
    outfile << "      { ";
    bool first = true;
    for (const auto cn2: cn2_table) {
      if (first) first = false;
      else outfile << ",\n        ";
      const double logCn2 = (double)std::log2(cn2);
      outfile << logCn2;
    }
    outfile << " },\n";

    // write the values of cn3, which are equal to cn2 + n
    outfile << "      { ";
    for (std::size_t i = std::size_t(0); i < VariableLog2ParamComplexityCTableNSize; ++i) {
      if (i > std::size_t(0)) outfile << ",\n        ";
      const double logCn3 = (double)std::log2(cn2_table[i] + i);
      outfile << logCn3;
    }
    outfile << " },\n";

    // write the values of cn4, which are equal to cn2 * (1 + n/2) + n
    outfile << "      { ";
    for (std::size_t i = std::size_t(0); i < VariableLog2ParamComplexityCTableNSize; ++i) {
      if (i > std::size_t(0)) outfile << ",\n        ";
      const double logCn4 = (double)std::log2(cn2_table[i] * (1.0 + i / 2.0) + i);
      outfile << logCn4;
    }
    outfile << " },\n";

    // write the values of cn5, which are equal to cn2 * (1 + 5n/6) + n + n^2/3
    outfile << "      { ";
    for (std::size_t i = std::size_t(0); i < VariableLog2ParamComplexityCTableNSize; ++i) {
      if (i > std::size_t(0)) outfile << ",\n        ";
      const double logCn5
          = (double)std::log2(cn2_table[i] * (1.0 + 5.0 * i / 6.0) + i + i * i / 3.0);
      outfile << logCn5;
    }
    outfile << " }\n";

    // write the footer and close the file
    outfile << "  };\n\n";
    outfile << "} /* namespace gum */\n";
    outfile.close();
  }

  // the CTable cache for log2(Cnr), n < 1000 and r in {2,3,4,5}
  const double VariableLog2ParamComplexityCTable[4][1000] = {{0,
                                                              1,
                                                              1.3219280948873624038,
                                                              1.5305148703230577745,
                                                              1.6865005853516363832,
                                                              1.8116353900270942834,
                                                              1.9163583844844809256,
                                                              2.0065271644294311315,
                                                              2.0857704966116465961,
                                                              2.1564986340920730967,
                                                              2.2203965250090043071,
                                                              2.2786902190668105739,
                                                              2.3322990996918790962,
                                                              2.3819333135364253451,
                                                              2.4281502505698502326,
                                                              2.4713970635580784219,
                                                              2.5120389433034655546,
                                                              2.5503761743106676541,
                                                              2.5866596296997594884,
                                                              2.6211016700234877064,
                                                              2.6538820078470202191,
                                                              2.6851559155058404293,
                                                              2.7150583520600020115,
                                                              2.7437053059117562093,
                                                              2.7711991437760832291,
                                                              2.79763156718914896,
                                                              2.8230804530606610214,
                                                              2.8476192305148533279,
                                                              2.8713103180418357852,
                                                              2.8942112722722521667,
                                                              2.9163733091011145859,
                                                              2.9378445477276624587,
                                                              2.9586657568594088019,
                                                              2.9788760126291720098,
                                                              2.9985108275544174461,
                                                              3.0176020678569406286,
                                                              3.0361794852617554064,
                                                              3.054270010943741287,
                                                              3.0718997050205953947,
                                                              3.0890904999196213687,
                                                              3.1058645897056522678,
                                                              3.1222418647242431433,
                                                              3.1382404560408532213,
                                                              3.1538784964406412215,
                                                              3.1691716096623245846,
                                                              3.1841343571506448562,
                                                              3.1987813380348359971,
                                                              3.2131256939662806715,
                                                              3.2271791995449077284,
                                                              3.2409529086398065623,
                                                              3.2544605816423333167,
                                                              3.2677107412971055922,
                                                              3.2807131309745982684,
                                                              3.2934769361591986758,
                                                              3.3060108350527364074,
                                                              3.3183230373618513021,
                                                              3.3304213181985549141,
                                                              3.3423130485925671707,
                                                              3.3540052232282806521,
                                                              3.365504485776890764,
                                                              3.3768171521001946189,
                                                              3.3879492315483643594,
                                                              3.3989064465446894303,
                                                              3.4096942506232235104,
                                                              3.4203178450640367991,
                                                              3.4307821942586769204,
                                                              3.4410920399192130859,
                                                              3.4512519142354931567,
                                                              3.4612661520716652142,
                                                              3.4711389022871026455,
                                                              3.4808741382541863452,
                                                              3.4904756676427037831,
                                                              3.4999471415291836252,
                                                              3.5092920628885035939,
                                                              3.5185137945163771356,
                                                              3.5276155664278530111,
                                                              3.5366004827734669469,
                                                              3.5454715283100086687,
                                                              3.5542315744600920802,
                                                              3.5628833849918311039,
                                                              3.5714296213456182549,
                                                              3.5798728476357393191,
                                                              3.5882155353493616623,
                                                              3.5964600677650082616,
                                                              3.6046087441100200799,
                                                              3.6126637834754928846,
                                                              3.6206273285060115441,
                                                              3.6285014488783944309,
                                                              3.6362881445853285634,
                                                              3.6439893490354289263,
                                                              3.6516069319827231254,
                                                              3.6591427022967493166,
                                                              3.6665984105827660322,
                                                              3.6739757516632027823,
                                                              3.6812763669277042133,
                                                              3.6885018465600203363,
                                                              3.6956537316512108049,
                                                              3.7027335162031129734,
                                                              3.709742649032294004,
                                                              3.7166825355772776796,
                                                              3.7235545396163005627,
                                                              3.7303599849017525791,
                                                              3.7371001567142370092,
                                                              3.7437763033413591351,
                                                              3.7503896374878165076,
                                                              3.7569413376170515129,
                                                              3.7634325492308957628,
                                                              3.769864386090051589,
                                                              3.776237931378247481,
                                                              3.7825542388137973759,
                                                              3.7888143337102198061,
                                                              3.7950192139907077404,
                                                              3.8011698511566636149,
                                                              3.8072671912138411621,
                                                              3.813312155558833183,
                                                              3.8193056418268742647,
                                                              3.8252485247035634686,
                                                              3.8311416567014151546,
                                                              3.8369858689048319533,
                                                              3.8427819716839515252,
                                                              3.8485307553783489887,
                                                              3.8542329909524455367,
                                                              3.8598894306260795872,
                                                              3.8655008084775124289,
                                                              3.8710678410228189783,
                                                              3.876591227773807713,
                                                              3.882071651772597054,
                                                              3.8875097801044717905,
                                                              3.8929062643933862731,
                                                              3.8982617412758329145,
                                                              3.9035768328586790687,
                                                              3.9088521471579533717,
                                                              3.9140882785229149654,
                                                              3.9192858080418018218,
                                                              3.9244453039363653701,
                                                              3.9295673219370383933,
                                                              3.9346524056491860577,
                                                              3.9397010869028807889,
                                                              3.9447138860910029301,
                                                              3.9496913124958692443,
                                                              3.9546338646043004417,
                                                              3.9595420304092820984,
                                                              3.9644162877048079174,
                                                              3.9692571043686029064,
                                                              3.9740649386334756521,
                                                              3.9788402393527797507,
                                                              3.9835834462536925393,
                                                              3.9882949901852819963,
                                                              3.9929752933534228276,
                                                              3.9976247695539166571,
                                                              4.0022438243928535329,
                                                              4.0068328555018473125,
                                                              4.0113922527467709145,
                                                              4.0159223984285752351,
                                                              4.0204236674779094685,
                                                              4.0248964276438972121,
                                                              4.0293410396769857584,
                                                              4.0337578575052832619,
                                                              4.0381472284054762412,
                                                              4.0425094931694394162,
                                                              4.0468449862648130377,
                                                              4.0511540359901010433,
                                                              4.0554369646269510241,
                                                              4.0596940885851129366,
                                                              4.0639257185455752719,
                                                              4.0681321595976314143,
                                                              4.0723137113727130298,
                                                              4.0764706681735560778,
                                                              4.0806033191015051997,
                                                              4.0847119481760802984,
                                                              4.0887968344559588729,
                                                              4.0928582521525260773,
                                                              4.096896470742770191,
                                                              4.1009117550762148596,
                                                              4.1049043654819339366,
                                                              4.1088745578704353179,
                                                              4.1128225838338998699,
                                                              4.1167486907416792619,
                                                              4.1206531218352182577,
                                                              4.1245361163190610299,
                                                              4.1283979094510820929,
                                                              4.132238732626686506,
                                                              4.1360588134663576199,
                                                              4.139858375893271436,
                                                              4.1436376402167542921,
                                                              4.1473968232080631324,
                                                              4.1511361381755982336,
                                                              4.1548557950390989646,
                                                              4.1585560004008677026,
                                                              4.1622369576153950277,
                                                              4.1658988668573400105,
                                                              4.1695419251878274025,
                                                              4.1731663266171183935,
                                                              4.1767722621704104213,
                                                              4.1803599199458023605,
                                                              4.1839294851754313953,
                                                              4.1874811402837543994,
                                                              4.1910150649428796754,
                                                              4.1945314361289192107,
                                                              4.1980304281754907691,
                                                              4.2015122128245518951,
                                                              4.2049769592809953522,
                                                              4.2084248342581513924,
                                                              4.2118560020307898029,
                                                              4.2152706244788999967,
                                                              4.2186688611367610946,
                                                              4.2220508692363925007,
                                                              4.225416803753091699,
                                                              4.2287668174469059679,
                                                              4.2321010609056335383,
                                                              4.2354196825864427467,
                                                              4.2387228288547360577,
                                                              4.2420106440234812695,
                                                              4.2452832703925196256,
                                                              4.2485408482849811307,
                                                              4.2517835160841324083,
                                                              4.2550114102685414608,
                                                              4.2582246654469715352,
                                                              4.26142341439412764,
                                                              4.2646077880808599403,
                                                              4.2677779157089421602,
                                                              4.2709339247424109587,
                                                              4.274075940938657503,
                                                              4.2772040883774593567,
                                                              4.2803184894933243854,
                                                              4.2834192651025713161,
                                                              4.2865065344308401762,
                                                              4.289580415144497394,
                                                              4.2926410233744620726,
                                                              4.2956884737434544164,
                                                              4.2987228793938232485,
                                                              4.3017443520103322285,
                                                              4.3047530018470325786,
                                                              4.3077489377515956193,
                                                              4.3107322671875394349,
                                                              4.3137030962596352168,
                                                              4.316661529734781233,
                                                              4.3196076710668069865,
                                                              4.3225416224157466871,
                                                              4.3254634846717516794,
                                                              4.3283733574735760996,
                                                              4.3312713392326411821,
                                                              4.3341575271484868992,
                                                              4.3370320172331373598,
                                                              4.3398949043284797966,
                                                              4.3427462821229729784,
                                                              4.3455862431756866471,
                                                              4.3484148789292014214,
                                                              4.3512322797302420696,
                                                              4.3540385348456256409,
                                                              4.356833732481850241,
                                                              4.3596179597991531196,
                                                              4.3623913029289527188,
                                                              4.3651538469910677875,
                                                              4.3679056761079779747,
                                                              4.3706468734213030913,
                                                              4.3733775211061569621,
                                                              4.3760977003871506241,
                                                              4.3788074915526937758,
                                                              4.3815069739687659833,
                                                              4.3841962260935583018,
                                                              4.3868753254904069294,
                                                              4.3895443488443355307,
                                                              4.3922033719697441967,
                                                              4.3948524698297672941,
                                                              4.3974917165443105915,
                                                              4.4001211854055455319,
                                                              4.4027409488877315979,
                                                              4.4053510786623730766,
                                                              4.4079516456074374631,
                                                              4.4105427198193112304,
                                                              4.413124370626925419,
                                                              4.4156966665987864573,
                                                              4.41825967555714616,
                                                              4.4208134645886669745,
                                                              4.4233581000526163152,
                                                              4.4258936475945764855,
                                                              4.4284201721547953312,
                                                              4.4309377379767873606,
                                                              4.4334464086213936085,
                                                              4.43594624697273332,
                                                              4.4384373152494838166,
                                                              4.4409196750140704779,
                                                              4.4433933871817323791,
                                                              4.4458585120283409253,
                                                              4.4483151092023449635,
                                                              4.45076323773105198,
                                                              4.453202956028516013,
                                                              4.455634321906808637,
                                                              4.4580573925828819171,
                                                              4.4604722246857066992,
                                                              4.4628788742656686495,
                                                              4.4652773968013761419,
                                                              4.4676678472091388983,
                                                              4.470050279849242969,
                                                              4.4724247485319761353,
                                                              4.4747913065293190016,
                                                              4.4771500065773404131,
                                                              4.479500900886822734,
                                                              4.4818440411483351227,
                                                              4.4841794785394766265,
                                                              4.486507263732785411,
                                                              4.4888274469003111022,
                                                              4.4911400777222940661,
                                                              4.4934452053927378401,
                                                              4.4957428786234023832,
                                                              4.4980331456558610981,
                                                              4.5003160542619937701,
                                                              4.5025916517526587413,
                                                              4.5048599849818993235,
                                                              4.5071211003542339668,
                                                              4.509375043832439367,
                                                              4.5116218609378160309,
                                                              4.5138615967603969992,
                                                              4.5160942959626275695,
                                                              4.5183200027852965519,
                                                              4.520538761050457488,
                                                              4.5227506141726685485,
                                                              4.5249556051553225799,
                                                              4.5271537766050604645,
                                                              4.5293451707288889807,
                                                              4.5315298293447279221,
                                                              4.5337077938817769152,
                                                              4.5358791053897657974,
                                                              4.5380438045398090452,
                                                              4.5402019316310466834,
                                                              4.5423535265940566674,
                                                              4.5444986289971485149,
                                                              4.5466372780479309412,
                                                              4.5487695126010194713,
                                                              4.5508953711591386693,
                                                              4.553014891878988557,
                                                              4.555128112576217525,
                                                              4.5572350707277848869,
                                                              4.5593358034746458429,
                                                              4.5614303476320641195,
                                                              4.5635187396852909814,
                                                              4.5656010157987418907,
                                                              4.5676772118182524807,
                                                              4.5697473632747893646,
                                                              4.5718115053883314758,
                                                              4.5738696730711936311,
                                                              4.5759219009318297111,
                                                              4.5779682232786962359,
                                                              4.580008674122619361,
                                                              4.5820432871812135645,
                                                              4.584072095882786968,
                                                              4.586095133367030563,
                                                              4.5881124324917292867,
                                                              4.5901240258338660283,
                                                              4.5921299456936059968,
                                                              4.5941302240962302861,
                                                              4.5961248927967330857,
                                                              4.5981139832828796798,
                                                              4.6000975267765156218,
                                                              4.6020755542377775882,
                                                              4.6040480963680820992,
                                                              4.6060151836140565962,
                                                              4.6079768461666379409,
                                                              4.6099331139685295611,
                                                              4.6118840167134669272,
                                                              4.6138295838511140801,
                                                              4.6157698445869090875,
                                                              4.6177048278901766665,
                                                              4.6196345624913410788,
                                                              4.6215590768852035097,
                                                              4.6234783993373831379,
                                                              4.6253925578819199416,
                                                              4.6273015803267174562,
                                                              4.6292054942556273289,
                                                              4.6311043270312808318,
                                                              4.6329981057947851042,
                                                              4.6348868574709847223,
                                                              4.6367706087697735384,
                                                              4.6386493861883799639,
                                                              4.6405232160142411146,
                                                              4.6423921243242238788,
                                                              4.6442561369929995507,
                                                              4.6461152796873097515,
                                                              4.6479695778763394642,
                                                              4.6498190568245965082,
                                                              4.6516637416049304576,
                                                              4.6535036570902770237,
                                                              4.6553388279622769375,
                                                              4.6571692787102758615,
                                                              4.6589950336343957105,
                                                              4.6608161168481601067,
                                                              4.6626325522775369237,
                                                              4.6644443636669574715,
                                                              4.6662515745784025611,
                                                              4.6680542083924478902,
                                                              4.6698522883139945705,
                                                              4.6716458373701978957,
                                                              4.6734348784128760812,
                                                              4.6752194341241404274,
                                                              4.6769995270109037122,
                                                              4.6787751794146723583,
                                                              4.6805464135071934706,
                                                              4.6823132512952057027,
                                                              4.6840757146187979032,
                                                              4.6858338251597526636,
                                                              4.6875876044357518424,
                                                              4.6893370738041744161,
                                                              4.6910822544682728719,
                                                              4.6928231674714568911,
                                                              4.6945598337046998694,
                                                              4.6962922739035599662,
                                                              4.6980205086537933923,
                                                              4.6997445583900265831,
                                                              4.701464443396743853,
                                                              4.7031801838139450922,
                                                              4.7048917996347849879,
                                                              4.7065993107051991018,
                                                              4.7083027367308956102,
                                                              4.7100020972750966664,
                                                              4.711697411761337051,
                                                              4.7133886994705447293,
                                                              4.7150759795503836003,
                                                              4.716759271010333876,
                                                              4.7184385927245848791,
                                                              4.7201139634314115412,
                                                              4.7217854017414575551,
                                                              4.7234529261286848367,
                                                              4.7251165549405174104,
                                                              4.7267763063949077562,
                                                              4.7284321985804709243,
                                                              4.7300842494594608212,
                                                              4.7317324768711976901,
                                                              4.7333768985298183551,
                                                              4.7350175320239840104,
                                                              4.7366543948237511685,
                                                              4.7382875042768928253,
                                                              4.7399168776121101132,
                                                              4.7415425319386947933,
                                                              4.7431644842495428449,
                                                              4.7447827514203240185,
                                                              4.7463973502112200009,
                                                              4.7480082972698944843,
                                                              4.7496156091286225731,
                                                              4.7512193022091739891,
                                                              4.7528193928204247598,
                                                              4.754415897162219018,
                                                              4.7560088313258255255,
                                                              4.7575982112920573996,
                                                              4.7591840529372824165,
                                                              4.7607663720297326293,
                                                              4.7623451842321564698,
                                                              4.7639205051033224336,
                                                              4.7654923501002564024,
                                                              4.7670607345735351856,
                                                              4.7686256737765431168,
                                                              4.7701871828583897184,
                                                              4.7717452768708836786,
                                                              4.7732999707639018894,
                                                              4.7748512793922586184,
                                                              4.7763992175118508143,
                                                              4.7779437997822711282,
                                                              4.779485040768117976,
                                                              4.7810229549378044922,
                                                              4.7825575566684097595,
                                                              4.7840888602415843067,
                                                              4.7856168798473426307,
                                                              4.7871416295831812349,
                                                              4.7886631234593268758,
                                                              4.7901813753921604899,
                                                              4.7916963992098375869,
                                                              4.793208208654144542,
                                                              4.7947168173766101518,
                                                              4.7962222389421578228,
                                                              4.7977244868320019222,
                                                              4.7992235744391740226,
                                                              4.8007195150724708554,
                                                              4.8022123219588612741,
                                                              4.8037020082385737396,
                                                              4.8051885869703383491,
                                                              4.806672071133410995,
                                                              4.8081524736236067596,
                                                              4.8096298072548462343,
                                                              4.8111040847635733186,
                                                              4.8125753188071112021,
                                                              4.8140435219620663077,
                                                              4.815508706729492161,
                                                              4.8169708855305968243,
                                                              4.8184300707133571606,
                                                              4.8198862745447037526,
                                                              4.8213395092214463844,
                                                              4.8227897868634492795,
                                                              4.8242371195151969587,
                                                              4.8256815191494624173,
                                                              4.8271229976646479187,
                                                              4.8285615668882222451,
                                                              4.8299972385755332027,
                                                              4.8314300244083812075,
                                                              4.8328599360013910768,
                                                              4.8342869848977274572,
                                                              4.8357111825698240182,
                                                              4.8371325404227860645,
                                                              4.8385510697928113544,
                                                              4.8399667819487799392,
                                                              4.8413796880902806308,
                                                              4.8427897993521877851,
                                                              4.8441971268038637177,
                                                              4.8456016814473992227,
                                                              4.8470034742173515596,
                                                              4.8484025159888686218,
                                                              4.849798817568856002,
                                                              4.8511923897029873842,
                                                              4.8525832430713089494,
                                                              4.853971388294493039,
                                                              4.8553568359266421339,
                                                              4.8567395964647177564,
                                                              4.8581196803414918861,
                                                              4.8594970979306104653,
                                                              4.8608718595434847742,
                                                              4.8622439754347617225,
                                                              4.8636134557955728042,
                                                              4.8649803107635145594,
                                                              4.8663445504125162344,
                                                              4.8677061847613245504,
                                                              4.8690652237710372319,
                                                              4.8704216773445008215,
                                                              4.8717755553306814065,
                                                              4.873126867518017491,
                                                              4.8744756236406097116,
                                                              4.8758218333816891743,
                                                              4.8771655063631165206,
                                                              4.878506652155493839,
                                                              4.8798452802753393698,
                                                              4.8811814001828057741,
                                                              4.8825150212892536317,
                                                              4.8838461529498840008,
                                                              4.8851748044667928639,
                                                              4.8865009850920388956,
                                                              4.8878247040260154321,
                                                              4.8891459704159974109,
                                                              4.8904647933599099119,
                                                              4.8917811819041414623,
                                                              4.8930951450436799277,
                                                              4.8944066917270454553,
                                                              4.8957158308505261957,
                                                              4.8970225712604733559,
                                                              4.8983269217598017775,
                                                              4.8996288910962872976,
                                                              4.9009284879733785445,
                                                              4.902225721046442608,
                                                              4.903520598923560847,
                                                              4.9048131301646282765,
                                                              4.9061033232843991314,
                                                              4.9073911867492512329,
                                                              4.9086767289830364192,
                                                              4.909959958360117227,
                                                              4.9112408832123994884,
                                                              4.9125195118262006133,
                                                              4.9137958524402511884,
                                                              4.9150699132533270941,
                                                              4.9163417024166076175,
                                                              4.9176112280417614286,
                                                              4.918878498192087001,
                                                              4.920143520891094191,
                                                              4.92140630411920732,
                                                              4.9226668558117179231,
                                                              4.9239251838645961001,
                                                              4.9251812961328695906,
                                                              4.9264352004250104855,
                                                              4.9276869045139148895,
                                                              4.9289364161274331622,
                                                              4.930183742953959225,
                                                              4.9314288926429705739,
                                                              4.9326718728017695526,
                                                              4.9339126909990707048,
                                                              4.9351513547632901435,
                                                              4.9363878715825793009,
                                                              4.9376222489086307732,
                                                              4.9388544941514433972,
                                                              4.9400846146855030838,
                                                              4.9413126178442805525,
                                                              4.9425385109245425497,
                                                              4.9437623011874967105,
                                                              4.9449839958525672046,
                                                              4.9462036021044877288,
                                                              4.9474211270911467864,
                                                              4.94863657792356193,
                                                              4.9498499616762590136,
                                                              4.951061285385411459,
                                                              4.9522705560567228389,
                                                              4.953477780653996021,
                                                              4.9546829661095603825,
                                                              4.9558861193187082606,
                                                              4.9570872471431259854,
                                                              4.9582863564077985785,
                                                              4.9594834539061531942,
                                                              4.9606785463950835435,
                                                              4.9618716405975016315,
                                                              4.9630627432033493918,
                                                              4.9642518608677059788,
                                                              4.9654390002135828652,
                                                              4.9666241678332916365,
                                                              4.9678073702783995813,
                                                              4.9689886140775181289,
                                                              4.9701679057200074396,
                                                              4.9713452516644425216,
                                                              4.9725206583386771797,
                                                              4.9736941321387124759,
                                                              4.9748656794271779447,
                                                              4.9760353065382254556,
                                                              4.9772030197691305986,
                                                              4.9783688253938667145,
                                                              4.9795327296494962255,
                                                              4.9806947387447246811,
                                                              4.9818548588592266313,
                                                              4.9830130961401488676,
                                                              4.9841694567042216235,
                                                              4.9853239466419916326,
                                                              4.9864765720105195257,
                                                              4.9876273388394878339,
                                                              4.9887762531291475199,
                                                              4.9899233208496038827,
                                                              4.991068547943197764,
                                                              4.9922119403239273439,
                                                              4.9933535038756051705,
                                                              4.9944932444557128548,
                                                              4.9956311678911315965,
                                                              4.9967672799847102638,
                                                              4.9979015865084877035,
                                                              4.9990340932062125034,
                                                              5.0001648057961816107,
                                                              5.0012937299703708049,
                                                              5.0024208713897557743,
                                                              5.0035462356939701678,
                                                              5.0046698284898649689,
                                                              5.005791655360565251,
                                                              5.0069117218661789437,
                                                              5.0080300335340695028,
                                                              5.0091465958702752204,
                                                              5.0102614143550079362,
                                                              5.011374494437898619,
                                                              5.0124858415485151752,
                                                              5.013595461088218741,
                                                              5.0147033584340050538,
                                                              5.0158095389358994254,
                                                              5.016914007923525709,
                                                              5.0180167706959730722,
                                                              5.019117832532923984,
                                                              5.0202171986848211915,
                                                              5.0213148743816304886,
                                                              5.0224108648275542777,
                                                              5.023505175204015849,
                                                              5.0245978106638364835,
                                                              5.0256887763431103977,
                                                              5.0267780773503911718,
                                                              5.0278657187677264773,
                                                              5.0289517056634220893,
                                                              5.0300360430708339621,
                                                              5.031118736011032766,
                                                              5.0321997894735872592,
                                                              5.033279208430878171,
                                                              5.0343569978304349988,
                                                              5.0354331625974140252,
                                                              5.0365077076369173525,
                                                              5.0375806378261662744,
                                                              5.038651958026854949,
                                                              5.0397216730759275549,
                                                              5.0407897877868261816,
                                                              5.041856306954672462,
                                                              5.0429212353508097166,
                                                              5.0439845777257694692,
                                                              5.0450463388084694216,
                                                              5.0461065233078148395,
                                                              5.0471651359122242653,
                                                              5.0482221812838998787,
                                                              5.0492776640733696425,
                                                              5.0503315889010202966,
                                                              5.0513839603731200967,
                                                              5.0524347830748155275,
                                                              5.0534840615674294639,
                                                              5.0545318003950736596,
                                                              5.05557800408043434,
                                                              5.0566226771290772035,
                                                              5.0576658240231813224,
                                                              5.0587074492262962266,
                                                              5.0597475571842567277,
                                                              5.0607861523217154698,
                                                              5.0618232390409545474,
                                                              5.0628588217326138121,
                                                              5.0638929047618450596,
                                                              5.0649254924753428497,
                                                              5.0659565892059488235,
                                                              5.0669861992626907821,
                                                              5.0680143269352670998,
                                                              5.0690409764981279039,
                                                              5.0700661522092937972,
                                                              5.0710898582985928229,
                                                              5.0721120989893737629,
                                                              5.0731328784784439634,
                                                              5.074152200949661129,
                                                              5.0751700705663855828,
                                                              5.0761864914751884115,
                                                              5.0772014678021966105,
                                                              5.0782150036598769915,
                                                              5.0792271031403100068,
                                                              5.0802377703232988182,
                                                              5.0812470092607417982,
                                                              5.0822548239972018536,
                                                              5.0832612185565020368,
                                                              5.0842661969462712435,
                                                              5.0852697631546215362,
                                                              5.0862719211552098741,
                                                              5.0872726749042893601,
                                                              5.0882720283436793096,
                                                              5.0892699853939786792,
                                                              5.0902665499607424593,
                                                              5.091261725937645366,
                                                              5.0922555171965226961,
                                                              5.0932479275947484254,
                                                              5.094238960975246755,
                                                              5.0952286211621018452,
                                                              5.0962169119655813532,
                                                              5.0972038371799763823,
                                                              5.0981894005837675721,
                                                              5.0991736059372865242,
                                                              5.1001564569885795564,
                                                              5.1011379574663200387,
                                                              5.1021181110905606815,
                                                              5.1030969215600734401,
                                                              5.1040743925584699525,
                                                              5.1050505277576778695,
                                                              5.1060253308124812222,
                                                              5.1069988053626742541,
                                                              5.1079709550334140289,
                                                              5.1089417834336536828,
                                                              5.1099112941604722948,
                                                              5.1108794907935513052,
                                                              5.1118463769002309149,
                                                              5.1128119560305220759,
                                                              5.11377623172347473,
                                                              5.1147392075002322542,
                                                              5.1157008868717728234,
                                                              5.116661273330823434,
                                                              5.1176203703575344761,
                                                              5.1185781814189557082,
                                                              5.1195347099664330059,
                                                              5.1204899594392658813,
                                                              5.1214439332609789091,
                                                              5.1223966348441702934,
                                                              5.1233480675841667207,
                                                              5.1242982348635015555,
                                                              5.1252471400564001414,
                                                              5.1261947865156995974,
                                                              5.1271411775841633229,
                                                              5.1280863165930874459,
                                                              5.1290302068593502938,
                                                              5.1299728516864773198,
                                                              5.1309142543640433587,
                                                              5.1318544181685155081,
                                                              5.1327933463647230639,
                                                              5.1337310422057331749,
                                                              5.1346675089254301128,
                                                              5.1356027497561580475,
                                                              5.1365367679072226892,
                                                              5.1374695665772396325,
                                                              5.1384011489590157851,
                                                              5.1393315182221570581,
                                                              5.1402606775344112933,
                                                              5.1411886300445432951,
                                                              5.1421153788897671078,
                                                              5.1430409271974797392,
                                                              5.1439652780826365941,
                                                              5.1448884346451242422,
                                                              5.1458103999755540059,
                                                              5.1467311771488537531,
                                                              5.1476507692358399737,
                                                              5.1485691792869845429,
                                                              5.1494864103439317304,
                                                              5.1504024654393774085,
                                                              5.1513173475900968512,
                                                              5.1522310598052882824,
                                                              5.1531436050768171242,
                                                              5.154054986390756099,
                                                              5.1549652067224593921,
                                                              5.1558742690289873778,
                                                              5.1567821762629959537,
                                                              5.1576889313596012698,
                                                              5.1585945372493107186,
                                                              5.1594989968480309273,
                                                              5.1604023130584497636,
                                                              5.1613044887765493485,
                                                              5.1622055268855175925,
                                                              5.1631054302586427696,
                                                              5.1640042017523919426,
                                                              5.1649018442226921621,
                                                              5.1657983605064501376,
                                                              5.1666937534335160009,
                                                              5.1675880258200974637,
                                                              5.1684811804786026102,
                                                              5.1693732202005282517,
                                                              5.1702641477748567667,
                                                              5.1711539659809906411,
                                                              5.1720426775792134322,
                                                              5.1729302853286380781,
                                                              5.1738167919714435072,
                                                              5.1747022002469185153,
                                                              5.1755865128781888274,
                                                              5.1764697325776172221,
                                                              5.1773518620504228593,
                                                              5.1782329039929742009,
                                                              5.1791128610875416527,
                                                              5.1799917360116936038,
                                                              5.1808695314259356479,
                                                              5.1817462499881834503,
                                                              5.1826218943436206388,
                                                              5.1834964671259511348,
                                                              5.1843699709595680858,
                                                              5.1852424084628720991,
                                                              5.1861137822415441789,
                                                              5.1869840948945533654,
                                                              5.1878533490036060627,
                                                              5.1887215471528209676,
                                                              5.1895886919050626673,
                                                              5.1904547858231815383,
                                                              5.1913198314537947198,
                                                              5.192183831339884037,
                                                              5.1930467880138593273,
                                                              5.1939087039925064815,
                                                              5.1947695817944223862,
                                                              5.1956294239197946538,
                                                              5.1964882328643193787,
                                                              5.1973460111126827954,
                                                              5.1982027611443841764,
                                                              5.1990584854245227575,
                                                              5.1999131864112939638,
                                                              5.2007668665594337654,
                                                              5.2016195283058488386,
                                                              5.2024711740834899132,
                                                              5.203321806317999787,
                                                              5.204171427421442786,
                                                              5.2050200398034256466,
                                                              5.2058676458613843963,
                                                              5.2067142479828039114,
                                                              5.2075598485492484713,
                                                              5.2084044499348749468,
                                                              5.2092480545023915894,
                                                              5.2100906646060316518,
                                                              5.2109322825951016611,
                                                              5.2117729108069843491,
                                                              5.2126125515733319205,
                                                              5.213451207214731653,
                                                              5.2142888800491773438,
                                                              5.2151255723779401663,
                                                              5.2159612865027060735,
                                                              5.2167960247110647387,
                                                              5.2176297892858354288,
                                                              5.2184625824991188736,
                                                              5.2192944066195696706,
                                                              5.220125263904654922,
                                                              5.2209551566026108915,
                                                              5.2217840869550187222,
                                                              5.2226120572001928366,
                                                              5.2234390695608432509,
                                                              5.2242651262570420911,
                                                              5.2250902295006715903,
                                                              5.2259143814924087224,
                                                              5.2267375844326826595,
                                                              5.2275598405055427875,
                                                              5.2283811518936920137,
                                                              5.2292015207701219026,
                                                              5.2300209492999023553,
                                                              5.2308394396413042671,
                                                              5.2316569939448518411,
                                                              5.232473614353325253,
                                                              5.2332893030042768601,
                                                              5.2341040620237198056,
                                                              5.2349178935364628629,
                                                              5.2357307996543065443,
                                                              5.2365427824845500737,
                                                              5.2373538441268099319,
                                                              5.238163986674397421,
                                                              5.2389732122109791135,
                                                              5.2397815228161856993,
                                                              5.2405889205596087876,
                                                              5.241395407508760762,
                                                              5.2422009857169236113,
                                                              5.2430056572353631594,
                                                              5.2438094241079955538,
                                                              5.2446122883720418528,
                                                              5.2454142520553670437,
                                                              5.2462153171821466202,
                                                              5.2470154857665374237,
                                                              5.2478147598175572952,
                                                              5.2486131413384571331,
                                                              5.2494106323246381152,
                                                              5.2502072347647210648,
                                                              5.2510029506425377477,
                                                              5.251797781929518294,
                                                              5.2525917305996054907,
                                                              5.2533847986118153983,
                                                              5.2541769879225519446,
                                                              5.2549683004819565113,
                                                              5.2557587382321022673,
                                                              5.2565483031086124299,
                                                              5.2573369970446170996,
                                                              5.2581248219619407536,
                                                              5.2589117797758726525,
                                                              5.2596978723983403015,
                                                              5.2604831017338673504,
                                                              5.2612674696825720844,
                                                              5.2620509781352522438,
                                                              5.2628336289762662759,
                                                              5.2636154240880941302,
                                                              5.2643963653400520641,
                                                              5.265176454600734246,
                                                              5.2659556937352318684,
                                                              5.2667340845913157565,
                                                              5.2675116290245780348,
                                                              5.2682883288747248685,
                                                              5.2690641859777169742,
                                                              5.2698392021680051656,
                                                              5.2706133792652787307,
                                                              5.271386719093205997,
                                                              5.2721592234625580531,
                                                              5.2729308941811661171,
                                                              5.2737017330491058331,
                                                              5.2744717418625182148,
                                                              5.2752409224120926368,
                                                              5.2760092764806643117,
                                                              5.2767768058460946534,
                                                              5.277543512281305027,
                                                              5.2783093975509522977,
                                                              5.2790744634183006667,
                                                              5.2798387116383533879,
                                                              5.2806021439611363633,
                                                              5.2813647621308099644,
                                                              5.2821265678829112389,
                                                              5.2828875629518661228,
                                                              5.2836477490664632839,
                                                              5.2844071279468405322,
                                                              5.2851657013085491243,
                                                              5.2859234708673588088,
                                                              5.2866804383222154584,
                                                              5.2874366053748973471,
                                                              5.2881919737240625778,
                                                              5.2889465450525703361,
                                                              5.2897003210486817082,
                                                              5.290453303389528017,
                                                              5.2912054937457764225,
                                                              5.2919568937911822815,
                                                              5.2927075051818022189,
                                                              5.293457329578695969,
                                                              5.2942063686346854112,
                                                              5.2949546239952445248,
                                                              5.2957020973001824871,
                                                              5.2964487901885801691,
                                                              5.2971947042925213722,
                                                              5.297939841235940861,
                                                              5.2986842026422600327,
                                                              5.2994277901252901941,
                                                              5.300170605297537918,
                                                              5.3009126497653964449,
                                                              5.3016539251279422018,
                                                              5.3023944329858592184,
                                                              5.3031341749252627338,
                                                              5.3038731525346189954,
                                                              5.3046113673943970923,
                                                              5.3053488210808819048,
                                                              5.306085515167051625,
                                                              5.3068214512167317665,
                                                              5.3075566307942354527,
                                                              5.30829105545386426,
                                                              5.3090247267513417384,
                                                              5.3097576462314437507,
                                                              5.3104898154367718988,
                                                              5.3112212359058306177,
                                                              5.3119519091693705448,
                                                              5.3126818367597845594,
                                                              5.3134110201977717836,
                                                              5.3141394610034575763,
                                                              5.3148671606899267061,
                                                              5.3155941207672805504,
                                                              5.3163203427447527361,
                                                              5.3170458281155896785,
                                                              5.317770578381833424,
                                                              5.3184945950302289219,
                                                              5.319217879550071082,
                                                              5.3199404334246933601,
                                                              5.3206622581300351271,
                                                              5.3213833551401314992,
                                                              5.3221037259237444772,
                                                              5.322823371944543247,
                                                              5.3235422946629959995,
                                                              5.3242604955346219953,
                                                              5.3249779760101221271,
                                                              5.3256947375405347955,
                                                              5.3264107815598338291,
                                                              5.3271261095145172604,
                                                              5.3278407228337982815,
                                                              5.3285546229469780144,
                                                              5.3292678112797853274,
                                                              5.3299802892558414413,
                                                              5.3306920582895092053,
                                                              5.3314031197943716478,
                                                              5.3321134751741725566},
                                                             {0,
                                                              1.5849625007211560757,
                                                              2.1699250014423121513,
                                                              2.5579955284837421203,
                                                              2.8517490673526251221,
                                                              3.0892269252300743609,
                                                              3.2890510328453164135,
                                                              3.4618085511600300563,
                                                              3.6141229290066028135,
                                                              3.7504259155236727707,
                                                              3.8738343594563144912,
                                                              3.9866268417936208124,
                                                              4.0905208835103650244,
                                                              4.1868449994830516658,
                                                              4.2766477253538210945,
                                                              4.3607719993149194337,
                                                              4.4399061203561185351,
                                                              4.5146190672093302609,
                                                              4.5853869701682743454,
                                                              4.6526125097847090828,
                                                              4.7166389444279692711,
                                                              4.7777617636723892502,
                                                              4.8362371518461326758,
                                                              4.8922882172546229285,
                                                              4.9461109414819546259,
                                                              4.9978786087594517795,
                                                              5.0477441915345888646,
                                                              5.0958449174691766714,
                                                              5.142302922386489783,
                                                              5.1872282989971658651,
                                                              5.2307202260764773527,
                                                              5.272868845842570984,
                                                              5.3137554441091534585,
                                                              5.3534544662673164694,
                                                              5.3920338893693333659,
                                                              5.4295558604443643347,
                                                              5.4660775512431785117,
                                                              5.5016515105679610897,
                                                              5.5363264478263589652,
                                                              5.5701470085010997835,
                                                              5.6031549010092156493,
                                                              5.6353887148233496873,
                                                              5.666884276405992793,
                                                              5.6976751795378968168,
                                                              5.7277925478387725633,
                                                              5.7572654681792094777,
                                                              5.786121332867177891,
                                                              5.8143857277800439221,
                                                              5.8420825800477009793,
                                                              5.8692343821621539846,
                                                              5.8958628449458272414,
                                                              5.9219877053718938953,
                                                              5.9476278617615392363,
                                                              5.9728011606503024211,
                                                              5.9975244751027325663,
                                                              6.0218137748042357416,
                                                              6.0456841897349296033,
                                                              6.0691500682567394165,
                                                              6.0922250302291720558,
                                                              6.1149220156687800198,
                                                              6.1372533293993125625,
                                                              6.1592306820850524218,
                                                              6.1808652279951372677,
                                                              6.2021675998074830005,
                                                              6.2231479407269647552,
                                                              6.2438159341637256006,
                                                              6.2641808311908375728,
                                                              6.2842514759783121292,
                                                              6.3040363293799472899,
                                                              6.3235434908322130099,
                                                              6.3427807187080356144,
                                                              6.3617554492549714951,
                                                              6.3804748142341640715,
                                                              6.3989456573660392635,
                                                              6.4171745496783119123,
                                                              6.4351678038431758822,
                                                              6.4529314875827097353,
                                                              6.4704714362143045392,
                                                              6.4877932644015752217,
                                                              6.5049023771705218877,
                                                              6.5218039802452860698,
                                                              6.5385030897536235983,
                                                              6.5550045413475777067,
                                                              6.5713129987812495258,
                                                              6.5874329619839846472,
                                                              6.6033687746642577565,
                                                              6.6191246314767182568,
                                                              6.6347045847820735887,
                                                              6.6501125510274698982,
                                                              6.6653523167725232668,
                                                              6.6804275443844689519,
                                                              6.6953417774240433502,
                                                              6.7100984457419530216,
                                                              6.7247008703045993983,
                                                              6.7391522677659985163,
                                                              6.7534557548017595252,
                                                              6.7676143522200646885,
                                                              6.7816309888629486835,
                                                              6.7955085053110009241,
                                                              6.8092496574029155454,
                                                              6.8228571195810641115,
                                                              6.8363334880734170085,
                                                              6.8496812839211305146,
                                                              6.8629029558607514971,
                                                              6.8760008830695991122,
                                                              6.8889773777816616374,
                                                              6.9018346877815446305,
                                                              6.9145749987831877093,
                                                              6.9272004366996293712,
                                                              6.9397130698098399293,
                                                              6.9521149108280493323,
                                                              6.9644079188810090741,
                                                              6.9765940013978378076,
                                                              6.9886750159171633356,
                                                              7.0006527718159281548,
                                                              7.0125290319637931802,
                                                              7.0243055143070174395,
                                                              7.0359838933852882903,
                                                              7.0475658017850904002,
                                                              7.0590528315326412923,
                                                              7.0704465354293475343,
                                                              7.0817484283326272987,
                                                              7.0929599883849858699,
                                                              7.1040826581934402029,
                                                              7.1151178459619242034,
                                                              7.1260669265790523852,
                                                              7.1369312426630076018,
                                                              7.1477121055656445137,
                                                              7.1584107963380390061,
                                                              7.1690285666587305613,
                                                              7.1795666397268513848,
                                                              7.1900262111213359972,
                                                              7.2004084496281084427,
                                                              7.2107144980362312126,
                                                              7.220945473905043599,
                                                              7.231102470302680274,
                                                              7.2411865565182020887,
                                                              7.2511987787476739342,
                                                              7.2611401607557306548,
                                                              7.2710117045136426484,
                                                              7.2808143908148288403,
                                                              7.2905491798685311267,
                                                              7.3002170118731619652,
                                                              7.3098188075696226562,
                                                              7.3193554687754387444,
                                                              7.3288278789009924097,
                                                              7.338236903447949544,
                                                              7.3475833904912182248,
                                                              7.3568681711443213445,
                                                              7.3660920600098922506,
                                                              7.3752558556148146707,
                                                              7.3843603408312903369,
                                                              7.393406283284258862,
                                                              7.4023944357455100373,
                                                              7.4113255365151546883,
                                                              7.4202003097909541296,
                                                              7.4290194660259221138,
                                                              7.4377837022745927342,
                                                              7.4464937025284365646,
                                                              7.4551501380409357367,
                                                              7.46375366764251158,
                                                              7.472304938045723155,
                                                              7.4808045841413086663,
                                                              7.489253229284988933,
                                                              7.4976514855758464861,
                                                              7.505999954126187923,
                                                              7.5142992253234304201,
                                                              7.5225498790841633934,
                                                              7.530752485100896898,
                                                              7.5389076030812161022,
                                                              7.5470157829804103145,
                                                              7.5550775652270116822,
                                                              7.563093480942076674,
                                                              7.5710640521519403379,
                                                              7.5789897919952027294,
                                                              7.5868712049236606276,
                                                              7.5947087868976961289,
                                                              7.6025030255760297493,
                                                              7.6102544005003744942,
                                                              7.617963383274865663,
                                                              7.6256304377407069239,
                                                              7.6332560201457235749,
                                                              7.640840579309796432,
                                                              7.6483845567851513891,
                                                              7.655888387013034091,
                                                              7.6633524974757074588,
                                                              7.670777308844529685,
                                                              7.6781632351241988488,
                                                              7.685510683793099318,
                                                              7.6928200559400004011,
                                                              7.7000917463971996213,
                                                              7.7073261438702260762,
                                                              7.7145236310640221689,
                                                              7.7216845848063044855,
                                                              7.7288093761672360671,
                                                              7.7358983705764634564,
                                                              7.7429519279370788709,
                                                              7.7499704027365883263,
                                                              7.7569541441553138128,
                                                              7.7639034961719657346,
                                                              7.7708187976664913066,
                                                              7.7777003825208090859,
                                                              7.784548579716417116,
                                                              7.7913637134303490583,
                                                              7.7981461031281220642,
                                                              7.8048960636549837844,
                                                              7.8116139053245801094,
                                                              7.8182999340057701687,
                                                              7.8249544512071169677,
                                                              7.8315777541595927858,
                                                              7.8381701358972941662,
                                                              7.844731885336115873,
                                                              7.8512632873507026687,
                                                              7.8577646228496798031,
                                                              7.8642361688489810234,
                                                              7.8706781985435876337,
                                                              7.8770909813775560337,
                                                              7.883474783112487394,
                                                              7.8898298658945753559,
                                                              7.896156488319749478,
                                                              7.9024549054977795137,
                                                              7.908725369114728565,
                                                              7.9149681274940411058,
                                                              7.9211834256561690637,
                                                              7.9273715053771969252,
                                                              7.9335326052458050583,
                                                              7.9396669607190686335,
                                                              7.9457748041773976766,
                                                              7.9518563649777309621,
                                                              7.9579118695058941313,
                                                              7.9639415412279186413,
                                                              7.9699456007399085422,
                                                              7.975924265817205594,
                                                              7.9818777514623482361,
                                                              7.9878062699518972423,
                                                              7.9937100308825455031,
                                                              7.999589241215965707,
                                                              8.0054441053230434022,
                                                              8.0112748250268044359,
                                                              8.0170815996448503427,
                                                              8.0228646260305200855,
                                                              8.0286240986136814257,
                                                              8.0343602094400452529,
                                                              8.0400731482104017545,
                                                              8.0457631023185527397,
                                                              8.0514302568884446032,
                                                              8.0570747948111662851,
                                                              8.0626968967803556154,
                                                              8.0682967413275274993,
                                                              8.0738745048563789197,
                                                              8.0794303616766942611,
                                                              8.0849644840372061338,
                                                              8.0904770421580760598,
                                                              8.0959682042627640897,
                                                              8.1014381366090564285,
                                                              8.1068870035196916035,
                                                              8.1123149674122601027,
                                                              8.1177221888286599238,
                                                              8.1231088264638859897,
                                                              8.1284750371942582348,
                                                              8.1338209761051807334,
                                                              8.1391467965182808797,
                                                              8.1444526500183069828,
                                                              8.1497386864789014282,
                                                              8.1550050540887468742,
                                                              8.1602518993764316946,
                                                              8.1654793672353918055,
                                                              8.170687600948014051,
                                                              8.1758767422096845223,
                                                              8.1810469311519824487,
                                                              8.1861983063656609261,
                                                              8.1913310049233842847,
                                                              8.1964451624015488562,
                                                              8.2015409129022316392,
                                                              8.2066183890745207918,
                                                              8.2116777221352794669,
                                                              8.2167190418899522797,
                                                              8.2217424767526345875,
                                                              8.2267481537658095903,
                                                              8.2317361986201333934,
                                                              8.2367067356733052463,
                                                              8.241659887969012388,
                                                              8.2465957772553721838,
                                                              8.2515145240030918217,
                                                              8.2564162474231999056,
                                                              8.2613010654847940373,
                                                              8.2661690949320902888,
                                                              8.2710204513012843819,
                                                              8.2758552489373915506,
                                                              8.2806736010104877721,
                                                              8.2854756195317040834,
                                                              8.2902614153691143173,
                                                              8.2950310982631840773,
                                                              8.2997847768421522119,
                                                              8.3045225586369397774,
                                                              8.3092445500957996529,
                                                              8.3139508565991242506,
                                                              8.3186415824733579427,
                                                              8.3233168310052789707,
                                                              8.3279767044556507471,
                                                              8.3326213040728056569,
                                                              8.3372507301060672091,
                                                              8.3418650818187281004,
                                                              8.3464644575010957794,
                                                              8.3510489544831312259,
                                                              8.3556186691467750904,
                                                              8.3601736969386344356,
                                                              8.3647141323816800451,
                                                              8.369240069087316769,
                                                              8.3737515997669760281,
                                                              8.3782488162437136481,
                                                              8.3827318094636957824,
                                                              8.3872006695069654114,
                                                              8.3916554855987435246,
                                                              8.3960963461200979197,
                                                              8.4005233386186137778,
                                                              8.4049365498186912049,
                                                              8.4093360656322566626,
                                                              8.4137219711682984524,
                                                              8.4180943507434804474,
                                                              8.4224532878914430967,
                                                              8.4267988653728433945,
                                                              8.4311311651845954884,
                                                              8.4354502685695358366,
                                                              8.4397562560254097974,
                                                              8.4440492073140998031,
                                                              8.4483292014705000383,
                                                              8.4525963168114621737,
                                                              8.4568506309442881275,
                                                              8.4610922207755105973,
                                                              8.4653211625191104872,
                                                              8.4695375317049190755,
                                                              8.473741403186858534,
                                                              8.4779328511508715849,
                                                              8.4821119491227694454,
                                                              8.4862787699764776761,
                                                              8.4904333859411753593,
                                                              8.4945758686092514012,
                                                              8.4987062889436693069,
                                                              8.5028247172853355096,
                                                              8.506931223360369998,
                                                              8.5110258762872401661,
                                                              8.5151087445838271606,
                                                              8.5191798961743927521,
                                                              8.5232393983963561368,
                                                              8.5272873180071080412,
                                                              8.5313237211906951529,
                                                              8.5353486735642061234,
                                                              8.5393622401844559988,
                                                              8.5433644855542283381,
                                                              8.5473554736285937139,
                                                              8.5513352678210345914,
                                                              8.5553039310096448133,
                                                              8.559261525543142568,
                                                              8.5632081132467057216,
                                                              8.5671437554278941917,
                                                              8.5710685128824284362,
                                                              8.5749824458999448495,
                                                              8.5788856142693692419,
                                                              8.5827780772847415136,
                                                              8.5866598937504541311,
                                                              8.5905311219867694916,
                                                              8.5943918198349464888,
                                                              8.5982420446628271549,
                                                              8.6020818533696523644,
                                                              8.6059113023911972817,
                                                              8.6097304477050542459,
                                                              8.6135393448352655099,
                                                              8.617338048857405397,
                                                              8.6211266144033853465,
                                                              8.6249050956662340894,
                                                              8.6286735464046238064,
                                                              8.6324320199476947124,
                                                              8.6361805691995492396,
                                                              8.6399192466437746418,
                                                              8.6436481043479265196,
                                                              8.6473671939677334564,
                                                              8.6510765667518167987,
                                                              8.6547762735454778493,
                                                              8.6584663647954265286,
                                                              8.6621468905533784977,
                                                              8.6658179004808459922,
                                                              8.6694794438525466518,
                                                              8.6731315695608337535,
                                                              8.676774326119449654,
                                                              8.6804077616675030526,
                                                              8.6840319239733609891,
                                                              8.6876468604382779404,
                                                              8.6912526181003944004,
                                                              8.6948492436382860404,
                                                              8.6984367833745608323,
                                                              8.7020152832797155185,
                                                              8.705584788975468058,
                                                              8.7091453457383103398,
                                                              8.7126969985032332033,
                                                              8.7162397918667089414,
                                                              8.719773770090577969,
                                                              8.7232989771050313266,
                                                              8.7268154565120994448,
                                                              8.7303232515887216891,
                                                              8.7338224052903772332,
                                                              8.7373129602538170957,
                                                              8.7407949588003539532,
                                                              8.7442684429392354417,
                                                              8.7477334543702767178,
                                                              8.751190034487249747,
                                                              8.7546382243806064594,
                                                              8.7580780648406779676,
                                                              8.7615095963604296969,
                                                              8.7649328591383266485,
                                                              8.7683478930814207075,
                                                              8.7717547378079583353,
                                                              8.7751534326500451044,
                                                              8.7785440166567312303,
                                                              8.7819265285965109058,
                                                              8.7853010069600987464,
                                                              8.7886674899628154378,
                                                              8.7920260155476928077,
                                                              8.7953766213876924951,
                                                              8.7987193448883527225,
                                                              8.8020542231902112462,
                                                              8.8053812931717150292,
                                                              8.80870059145107831,
                                                              8.8120121543892508953,
                                                              8.8153160180920835387,
                                                              8.8186122184126460866,
                                                              8.8219007909536948375,
                                                              8.825181771070118586,
                                                              8.8284551938710524865,
                                                              8.8317210942220629732,
                                                              8.8349795067477270294,
                                                              8.8382304658335790748,
                                                              8.8414740056284273351,
                                                              8.8447101600464375082,
                                                              8.8479389627694029485,
                                                              8.851160447248741292,
                                                              8.8543746467076420714,
                                                              8.85758159414324453,
                                                              8.8607813223284672688,
                                                              8.8639738638142553384,
                                                              8.8671592509313903463,
                                                              8.8703375157926309669,
                                                              8.8735086902946509468,
                                                              8.8766728061198012512,
                                                              8.87982989473832518,
                                                              8.8829799874099784063,
                                                              8.8861231151860042843,
                                                              8.8892593089110185645,
                                                              8.8923885992249100951,
                                                              8.8955110165643418441,
                                                              8.8986265911650068716,
                                                              8.9017353530629570457,
                                                              8.9048373320966920375,
                                                              8.9079325579085857356,
                                                              8.9110210599469397152,
                                                              8.9141028674674096521,
                                                              8.9171780095347799033,
                                                              8.9202465150246617043,
                                                              8.9233084126249995194,
                                                              8.9263637308379379931,
                                                              8.9294124979811471121,
                                                              8.9324547421895719168,
                                                              8.9354904914169157593,
                                                              8.9385197734374433054,
                                                              8.9415426158471085216,
                                                              8.9445590460653381371,
                                                              8.9475690913366126011,
                                                              8.9505727787316828881,
                                                              8.9535701351492082978,
                                                              8.9565611873173409663,
                                                              8.9595459617948716158,
                                                              8.962524484972838934,
                                                              8.9654967830760323722,
                                                              8.9684628821640810514,
                                                              8.9714228081330880116,
                                                              8.9743765867170655071,
                                                              8.9773242434890434538,
                                                              8.9802658038624407766,
                                                              8.9832012930926143923,
                                                              8.9861307362780333818,
                                                              8.9890541583614531618,
                                                              8.9919715841314573623,
                                                              8.994883038223447258,
                                                              8.9977885451212387125,
                                                              9.0006881291578313409,
                                                              9.0035818145172239468,
                                                              9.0064696252352121064,
                                                              9.0093515852006547107,
                                                              9.0122277181568346549,
                                                              9.0150980477024660331,
                                                              9.0179625972930210764,
                                                              9.020821390241794191,
                                                              9.0236744497209304683,
                                                              9.0265217987628947327,
                                                              9.029363460261317087,
                                                              9.0321994569721244517,
                                                              9.0350298115147893441,
                                                              9.0378545463733157561,
                                                              9.0406736838973671411,
                                                              9.0434872463032078826,
                                                              9.0462952556749733901,
                                                              9.0490977339656488709,
                                                              9.0518947029979894836,
                                                              9.0546861844654991103,
                                                              9.0574721999338638767,
                                                              9.0602527708414495322,
                                                              9.0630279185006585863,
                                                              9.065797664098649733,
                                                              9.0685620286986239336,
                                                              9.071321033240385745,
                                                              9.0740746985417786163,
                                                              9.0768230452992337831,
                                                              9.0795660940889533208,
                                                              9.0823038653676579912,
                                                              9.0850363794737685197,
                                                              9.0877636566279367258,
                                                              9.0904857169344612799,
                                                              9.093202580381621658,
                                                              9.0959142668429873169,
                                                              9.0986207960781442239,
                                                              9.1013221877335137577,
                                                              9.1040184613434220751,
                                                              9.1067096363305779505,
                                                              9.1093957320072203032,
                                                              9.1120767675761324966,
                                                              9.1147527621308750412,
                                                              9.1174237346571320728,
                                                              9.1200897040333597232,
                                                              9.1227506890314327137,
                                                              9.1254067083178451725,
                                                              9.1280577804540730114,
                                                              9.1307039238975100659,
                                                              9.1333451570023846955,
                                                              9.1359814980204188117,
                                                              9.1386129651014957886,
                                                              9.1412395762946001554,
                                                              9.1438613495484197813,
                                                              9.1464783027120866166,
                                                              9.1490904535361448069,
                                                              9.1516978196729557027,
                                                              9.1543004186775274178,
                                                              9.1568982680085433401,
                                                              9.1594913850284438439,
                                                              9.1620797870046715161,
                                                              9.1646634911101561016,
                                                              9.1672425144240268224,
                                                              9.169816873932221668,
                                                              9.1723865865283151777,
                                                              9.1749516690139980568,
                                                              9.1775121381000293042,
                                                              9.1800680104065133236,
                                                              9.1826193024639000129,
                                                              9.1851660307134590511,
                                                              9.1877082115077879365,
                                                              9.1902458611118387211,
                                                              9.1927789957031240675,
                                                              9.195307631372749313,
                                                              9.1978317841255563536,
                                                              9.2003514698811557082,
                                                              9.202866704474359949,
                                                              9.2053775036556562128,
                                                              9.2078838830920837211,
                                                              9.2103858583677364891,
                                                              9.2128834449840368848,
                                                              9.2153766583608636154,
                                                              9.2178655138366352162,
                                                              9.2203500266691449383,
                                                              9.2228302120361433936,
                                                              9.2253060850357151423,
                                                              9.227777660687003447,
                                                              9.2302449539306632431,
                                                              9.2327079796293904934,
                                                              9.2351667525686451654,
                                                              9.2376212874569070266,
                                                              9.2400715989265123085,
                                                              9.2425177015338793041,
                                                              9.2449596097602455558,
                                                              9.2473973380122238552,
                                                              9.2498309006219940898,
                                                              9.2522603118481541173,
                                                              9.254685585876105236,
                                                              9.2571067368185335766,
                                                              9.2595237787159199172,
                                                              9.2619367255369269287,
                                                              9.2643455911792624846,
                                                              9.2667503894695819611,
                                                              9.2691511341644829969,
                                                              9.2715478389506813528,
                                                              9.2739405174456504,
                                                              9.2763291831979213242,
                                                              9.2787138496877830107,
                                                              9.2810945303274987594,
                                                              9.2834712384618587322,
                                                              9.285843987368689767,
                                                              9.2882127902591804514,
                                                              9.2905776602784708729,
                                                              9.2929386105061357881,
                                                              9.2952956539561206739,
                                                              9.2976488035780473496,
                                                              9.2999980722568178493,
                                                              9.3023434728134599681,
                                                              9.3046850180055340473,
                                                              9.3070227205274900228,
                                                              9.3093565930109996032,
                                                              9.311686648025593982,
                                                              9.3140128980786549562,
                                                              9.3163353556164789637,
                                                              9.3186540330239928664,
                                                              9.320968942625574627,
                                                              9.3232800966853961455,
                                                              9.3255875074076399756,
                                                              9.3278911869369842691,
                                                              9.3301911473591836454,
                                                              9.3324874007010816257,
                                                              9.3347799589312909774,
                                                              9.3370688339604512862,
                                                              9.3393540376415664639,
                                                              9.34163558177047193,
                                                              9.3439134780861792251,
                                                              9.3461877382711318063,
                                                              9.3484583739517486123,
                                                              9.350725396698571501,
                                                              9.3529888180269242781,
                                                              9.3552486493969340131,
                                                              9.3575049022139946686,
                                                              9.3597575878292413876,
                                                              9.3620067175398471448,
                                                              9.3642523025891293287,
                                                              9.3664943541673526539,
                                                              9.3687328834114964593,
                                                              9.3709679014060256463,
                                                              9.3731994191832459506,
                                                              9.375427447723154728,
                                                              9.3776519979543113692,
                                                              9.3798730807539385523,
                                                              9.3820907069480021789,
                                                              9.3843048873120320508,
                                                              9.3865156325710366048,
                                                              9.3887229533999985165,
                                                              9.3909268604240470069,
                                                              9.393127364219079567,
                                                              9.3953244753115736643,
                                                              9.3975182041794180776,
                                                              9.3997085612517317088,
                                                              9.4018955569095847835,
                                                              9.4040792014860254966,
                                                              9.4062595052665187723,
                                                              9.4084364784889498168,
                                                              9.4106101313444767698,
                                                              9.412780473977290896,
                                                              9.4149475164849523168,
                                                              9.4171112689192728595,
                                                              9.4192717412855753167,
                                                              9.4214289435440576881,
                                                              9.4235828856091341521,
                                                              9.4257335773503960752,
                                                              9.4278810285925107593,
                                                              9.4300252491156033585,
                                                              9.4321662486556316907,
                                                              9.4343040369042689974,
                                                              9.4364386235097583722,
                                                              9.4385700180767280187,
                                                              9.440698230166502114,
                                                              9.4428232692976035167,
                                                              9.4449451449457448859,
                                                              9.4470638665442159265,
                                                              9.4491794434840983286,
                                                              9.4512918851145837351,
                                                              9.4534012007431940106,
                                                              9.4555073996357528188,
                                                              9.4576104910173146578,
                                                              9.4597104840715289242,
                                                              9.4618073879416328964,
                                                              9.4639012117303611404,
                                                              9.4659919645000414334,
                                                              9.4680796552731454341,
                                                              9.4701642930323188807,
                                                              9.472245886720807917,
                                                              9.4743244452423986957,
                                                              9.4763999774618543626,
                                                              9.4784724922051921681,
                                                              9.4805419982597332051,
                                                              9.4826085043742764924,
                                                              9.4846720192598077404,
                                                              9.4867325515890765786,
                                                              9.4887901099971383445,
                                                              9.4908447030817804091,
                                                              9.4928963394032717105,
                                                              9.4949450274847819742,
                                                              9.4969907758127760644,
                                                              9.4990335928371791852,
                                                              9.5010734869710180561,
                                                              9.503110466591646599,
                                                              9.5051445400400176311,
                                                              9.507175715621507095,
                                                              9.5092040016057683971,
                                                              9.5112294062271054429,
                                                              9.5132519376844975056,
                                                              9.5152716041421108173,
                                                              9.5172884137291759998,
                                                              9.5193023745405671576,
                                                              9.5213134946362618649,
                                                              9.5233217820424851396,
                                                              9.5253272447512795651,
                                                              9.5273298907209031938,
                                                              9.5293297278758579694,
                                                              9.5313267641073178282,
                                                              9.5333210072731695561,
                                                              9.5353124651983272031,
                                                              9.537301145674614844,
                                                              9.5392870564612177731,
                                                              9.5412702052850129064,
                                                              9.5432505998402863412,
                                                              9.5452282477892520518,
                                                              9.5472031567622206438,
                                                              9.5491753343575780377,
                                                              9.5511447881421851491,
                                                              9.5531115256514489431,
                                                              9.5550755543895036226,
                                                              9.5570368818292728008,
                                                              9.5589955154129011561,
                                                              9.560951462551601665,
                                                              9.5629047306263057493,
                                                              9.5648553269872600424,
                                                              9.566803258954459821,
                                                              9.5687485338179740779,
                                                              9.5706911588378549283,
                                                              9.5726311412443987336,
                                                              9.5745684882383184089,
                                                              9.5765032069908322399,
                                                              9.5784353046440138257,
                                                              9.5803647883107014849,
                                                              9.5822916650748783951,
                                                              9.5842159419916015395,
                                                              9.5861376260874422428,
                                                              9.5880567243603245231,
                                                              9.5899732437800206952,
                                                              9.5918871912879453134,
                                                              9.593798573797551299,
                                                              9.5957073981944507324,
                                                              9.5976136713364592623,
                                                              9.5995174000538874282,
                                                              9.6014185911495335546,
                                                              9.6033172513990301411,
                                                              9.605213387550714188,
                                                              9.6071070063259647043,
                                                              9.6089981144195437679,
                                                              9.6108867184991506605,
                                                              9.6127728252060666847,
                                                              9.6146564411552315477,
                                                              9.6165375729352522427,
                                                              9.6184162271085789087,
                                                              9.6202924102116149641,
                                                              9.6221661287548982955,
                                                              9.6240373892232859987,
                                                              9.6259061980760947108,
                                                              9.6277725617468910002,
                                                              9.6296364866444132957,
                                                              9.6314979791517441043,
                                                              9.6333570456269885796,
                                                              9.6352136924036173582,
                                                              9.637067925789827072,
                                                              9.638919752069607938,
                                                              9.6407691775021095992,
                                                              9.6426162083221047538,
                                                              9.6444608507401774489,
                                                              9.6463031109427408438,
                                                              9.6481429950920354344,
                                                              9.6499805093265145217,
                                                              9.6518156597606754588,
                                                              9.6536484524856920331,
                                                              9.6554788935688211637,
                                                              9.6573069890540246263,
                                                              9.6591327449620880685,
                                                              9.660956167290423835,
                                                              9.6627772620135541359,
                                                              9.6645960350828108432,
                                                              9.6664124924269376748,
                                                              9.6682266399520031541,
                                                              9.6700384835411750117,
                                                              9.6718480290554449397,
                                                              9.6736552823331614093,
                                                              9.6754602491907100159,
                                                              9.6772629354221866294,
                                                              9.6790633467995643713,
                                                              9.6808614890730950719,
                                                              9.6826573679711405163,
                                                              9.6844509892004175811,
                                                              9.6862423584457957304,
                                                              9.6880314813711176924,
                                                              9.6898183636184977985,
                                                              9.6916030108089490369,
                                                              9.6933854285422000885,
                                                              9.695165622397226457,
                                                              9.6969435979316163099,
                                                              9.6987193606824604331,
                                                              9.7004929161661550552,
                                                              9.7022642698780909853,
                                                              9.7040334272935258042,
                                                              9.7058003938670083244,
                                                              9.7075651750330749223,
                                                              9.709327776205789462,
                                                              9.7110882027790612625,
                                                              9.7128464601269044465,
                                                              9.714602553603462809,
                                                              9.7163564885428765905,
                                                              9.71810827025978341,
                                                              9.7198579040488848335,
                                                              9.7216053951856586934,
                                                              9.7233507489260198042,
                                                              9.7250939705064674001,
                                                              9.7268350651442680999,
                                                              9.7285740380376939385,
                                                              9.7303108943658269681,
                                                              9.732045639288898542,
                                                              9.7337782779479429252,
                                                              9.735508815465712118,
                                                              9.7372372569457716907,
                                                              9.7389636074734795557,
                                                              9.7406878721153962175,
                                                              9.7424100559198993921,
                                                              9.7441301639169832782,
                                                              9.7458482011181271076,
                                                              9.7475641725171211505,
                                                              9.749278083089297553,
                                                              9.7509899377922018004,
                                                              9.7526997415654061996,
                                                              9.7544074993308349519,
                                                              9.7561132159924639495,
                                                              9.7578168964367151261,
                                                              9.7595185455326891599,
                                                              9.761218168131639672,
                                                              9.7629157690676375836,
                                                              9.7646113531575178257,
                                                              9.7663049252006874923,
                                                              9.7679964899796782873,
                                                              9.7696860522598232279,
                                                              9.77137361678941474,
                                                              9.7730591882999711117,
                                                              9.7747427715062382703,
                                                              9.7764243711061045161,
                                                              9.7781039917808012518,
                                                              9.7797816381951268028,
                                                              9.7814573149972368071,
                                                              9.7831310268189692891,
                                                              9.7848027782757078796,
                                                              9.7864725739668010362,
                                                              9.7881404184750540054,
                                                              9.7898063163675157483,
                                                              9.7914702721949478104,
                                                              9.7931322904922861738,
                                                              9.7947923757784653986,
                                                              9.7964505325568023153,
                                                              9.7981067653147295715,
                                                              9.799761078523964386,
                                                              9.8014134766406790789,
                                                              9.8030639641057177869,
                                                              9.8047125453441488219,
                                                              9.8063592247658739609,
                                                              9.808004006765477456,
                                                              9.8096468957221532037,
                                                              9.8112878960002429807,
                                                              9.8129270119486307067,
                                                              9.8145642479014316706,
                                                              9.8161996081777154188,
                                                              9.8178330970816496404,
                                                              9.8194647189026085243,
                                                              9.8210944779152047346,
                                                              9.8227223783793551348,
                                                              9.8243484245404442134,
                                                              9.8259726206291411188,
                                                              9.8275949708618810519,
                                                              9.8292154794404424933,
                                                              9.8308341505523735293,
                                                              9.8324509883709101388,
                                                              9.8340659970551094204,
                                                              9.8356791807497643276,
                                                              9.8372905435857056489,
                                                              9.8389000896796154905,
                                                              9.840507823134412746,
                                                              9.842113748038819665,
                                                              9.8437178684679249585,
                                                              9.8453201884830239266,
                                                              9.8469207121317090525,
                                                              9.8485194434478149361,
                                                              9.850116386451720274,
                                                              9.8517115451501400258,
                                                              9.853304923536390092,
                                                              9.8548965255904139582,
                                                              9.8564863552787667089,
                                                              9.8580744165547073976,
                                                              9.859660713358344708,
                                                              9.8612452496163776061,
                                                              9.8628280292427703557,
                                                              9.8644090561380899373,
                                                              9.8659883341900709297,
                                                              9.8675658672735178101,
                                                              9.8691416592502925198,
                                                              9.8707157139694334802,
                                                              9.8722880352673758608,
                                                              9.873858626967640717,
                                                              9.8754274928810925616,
                                                              9.8769946368061170006,
                                                              9.8785600625285159282,
                                                              9.8801237738216762807,
                                                              9.8816857744464314806,
                                                              9.883246068151228414,
                                                              9.8848046586723690154,
                                                              9.8863615497335946003,
                                                              9.8879167450466596279,
                                                              9.8894702483112570945,
                                                              9.8910220632145193775,
                                                              9.8925721934320840489,
                                                              9.894120642627184381,
                                                              9.8956674144512000169,
                                                              9.8972125125437848681,
                                                              9.8987559405323946038,
                                                              9.9002977020330895641,
                                                              9.9018378006498988242,
                                                              9.9033762399752784944,
                                                              9.9049130235899642827,
                                                              9.9064481550631775519,
                                                              9.9079816379526075565,
                                                              9.9095134758043652567,
                                                              9.9110436721531538495,
                                                              9.9125722305223096242,
                                                              9.9140991544237220268,
                                                              9.9156244473581569565,
                                                              9.9171481128150240636,
                                                              9.91867015427256149,
                                                              9.9201905751978429748,
                                                              9.9217093790467156822,
                                                              9.9232265692641501431,
                                                              9.9247421492840981472,
                                                              9.9262561225294287937,
                                                              9.9277684924121238907,
                                                              9.9292792623335266455,
                                                              9.9307884356836986228,
                                                              9.9322960158422564092,
                                                              9.9338020061781726611,
                                                              9.9353064100493160282,
                                                              9.936809230803342885,
                                                              9.9383104717770827108,
                                                              9.9398101362967672401,
                                                              9.9413082276784496827,
                                                              9.9428047492272302321,
                                                              9.9442997042381904294,
                                                              9.9457930959959259809,
                                                              9.9472849277746036023,
                                                              9.9487752028380889158,
                                                              9.9502639244401756002,
                                                              9.951751095824379334,
                                                              9.9532367202239377946,
                                                              9.9547208008621499431,
                                                              9.9562033409520598326,
                                                              9.9576843436968260903,
                                                              9.9591638122895727037,
                                                              9.9606417499133783622,
                                                              9.9621181597416814668,
                                                              9.9635930449376655105,
                                                              9.9650664086549767262,
                                                              9.9665382540373581577,
                                                              9.9680085842188272949,
                                                              9.9694774023237648919,
                                                              9.9709447114667177914,
                                                              9.972410514752809263,
                                                              9.9738748152773819555,
                                                              9.9753376161264686317,
                                                              9.9767989203763605133,
                                                              9.9782587310939732106,
                                                              9.9797170513368271827,
                                                              9.981173884152926945,
                                                              9.982629232581176737,
                                                              9.9840830996508902473,
                                                              9.9855354883822684542,
                                                              9.9869864017861758043,
                                                              9.9884358428643462702,
                                                              9.9898838146094934842,
                                                              9.9913303200047582919,
                                                              9.9927753620247212751,
                                                              9.994218943634486152,
                                                              9.9956610677904205176,
                                                              9.9971017374398591926,
                                                              9.9985409555210758015,
                                                              9.9999787249635438968,
                                                              10.001415048687835707,
                                                              10.002849929605661217,
                                                              10.004283370619978299,
                                                              10.005715374624967851,
                                                              10.007145944506056878,
                                                              10.008575083140170747,
                                                              10.010002793395079479,
                                                              10.011429078130582582,
                                                              10.012853940197395275,
                                                              10.014277382437839492,
                                                              10.015699407685753286,
                                                              10.017120018766567213,
                                                              10.018539218497068077,
                                                              10.019957009685757754,
                                                              10.021373395132535222},
                                                             {0,
                                                              2,
                                                              2.8073549220576041741,
                                                              3.3536370631535943332,
                                                              3.7714895106311274731,
                                                              4.1115655159417610065,
                                                              4.3990864632756769836,
                                                              4.6485587228878513244,
                                                              4.8691354356993361563,
                                                              5.0669818295080517601,
                                                              5.2464571207588184265,
                                                              5.4107623502682091754,
                                                              5.5623190195081955167,
                                                              5.703006746365687718,
                                                              5.8343126340271016517,
                                                              5.9574352031222339576,
                                                              6.0733556433332545055,
                                                              6.1828862241989339665,
                                                              6.2867078798528943651,
                                                              6.3853976809854211538,
                                                              6.4794477668063832354,
                                                              6.5692827294485240941,
                                                              6.6552714096263994037,
                                                              6.7377346850569788117,
                                                              6.8169550088563353185,
                                                              6.8931830556583326697,
                                                              6.9666387250830359434,
                                                              7.0375214021316967461,
                                                              7.1060071323467326465,
                                                              7.1722554489534200428,
                                                              7.2364098410621791047,
                                                              7.2986017269949030606,
                                                              7.3589482121966680594,
                                                              7.4175576159293372669,
                                                              7.4745288259800064523,
                                                              7.5299519580007014952,
                                                              7.5839102229344481643,
                                                              7.636479902408856546,
                                                              7.6877324099684969383,
                                                              7.7377320545346357505,
                                                              7.7865399808308852769,
                                                              7.8342124006415465587,
                                                              7.8808013171713264811,
                                                              7.9263562226457615623,
                                                              7.9709222592914628081,
                                                              8.014541602938578535,
                                                              8.0572545500522494422,
                                                              8.0990984438340216656,
                                                              8.1401078875996137896,
                                                              8.1803154063567138365,
                                                              8.2197544148135897046,
                                                              8.2584527204338957773,
                                                              8.296437980622632935,
                                                              8.3337363124671615111,
                                                              8.3703724126775718162,
                                                              8.4063696607852769205,
                                                              8.4417502120744956073,
                                                              8.4765350821312299701,
                                                              8.5107442240840303072,
                                                              8.5443965993439707773,
                                                              8.5775102425159595043,
                                                              8.6101023210584628487,
                                                              8.6421891902008951547,
                                                              8.6737864435672200614,
                                                              8.704908959903193022,
                                                              8.7355709462653319264,
                                                              8.7657859779877131956,
                                                              8.7955670357126791714,
                                                              8.824926539740017617,
                                                              8.8538763819264332255,
                                                              8.8824279553406579879,
                                                              8.9105921818632936038,
                                                              8.9383795378983084134,
                                                              8.9658000783512310505,
                                                              8.9928634590117084713,
                                                              9.019578957466292124,
                                                              9.0459554926564464949,
                                                              9.0720016431856116412,
                                                              9.0977256644704276312,
                                                              9.123135504822986519,
                                                              9.1482388205421223404,
                                                              9.1730429900877190619,
                                                              9.1975551274032909532,
                                                              9.2217820944479775847,
                                                              9.2457305129933153864,
                                                              9.2694067757362059723,
                                                              9.29281705677560943,
                                                              9.3159673214952327669,
                                                              9.3388633358935919659,
                                                              9.3615106753968824194,
                                                              9.3839147331894565696,
                                                              9.406080728093311194,
                                                              9.4280137120249349891,
                                                              9.449718577057726776,
                                                              9.4712000621136365197,
                                                              9.4924627593071306819,
                                                              9.5135111199645425728,
                                                              9.5343494603360063877,
                                                              9.5549819670216145084,
                                                              9.5754127021261776775,
                                                              9.5956456081599235119,
                                                              9.6156845127008168106,
                                                              9.6355331328306839112,
                                                              9.6551950793585383792,
                                                              9.674673860845226514,
                                                              9.6939728874374058165,
                                                              9.7130954745238167192,
                                                              9.7320448462231041731,
                                                              9.7508241387121277199,
                                                              9.7694364034039420375,
                                                              9.7878846099825516092,
                                                              9.8061716493038666442,
                                                              9.8243003361679885899,
                                                              9.8422734119705577172,
                                                              9.8600935472398969495,
                                                              9.877763344064923956,
                                                              9.8952853384199102038,
                                                              9.9126620023905385182,
                                                              9.9298957463077943686,
                                                              9.9469889207933022135,
                                                              9.963943818719975809,
                                                              9.9807626770924713355,
                                                              9.9974476788531525528,
                                                              10.014000954613710093,
                                                              10.030424584318293313,
                                                              10.046720598842311389,
                                                              10.062890981527374379,
                                                              10.078937669655852361,
                                                              10.094862555870758314,
                                                              10.110667489539036268,
                                                              10.126354278064880532,
                                                              10.141924688152043288,
                                                              10.157380447020491587,
                                                              10.172723243575690688,
                                                              10.187954729538160947,
                                                              10.203076520527410764,
                                                              10.218090197110742068,
                                                              10.2329973058114021,
                                                              10.247799360081446096,
                                                              10.262497841240575625,
                                                              10.277094199381865636,
                                                              10.291589854243699875,
                                                              10.305986196054695014,
                                                              10.320284586347755251,
                                                              10.334486358744756629,
                                                              10.348592819717516988,
                                                              10.362605249320205658,
                                                              10.376524901900120668,
                                                              10.390353006780426171,
                                                              10.404090768925623678,
                                                              10.417739369580614195,
                                                              10.431299966890730246,
                                                              10.444773696502085869,
                                                              10.458161672141519816,
                                                              10.471464986178167678,
                                                              10.484684710167483601,
                                                              10.497821895378095292,
                                                              10.510877573301435461,
                                                              10.523852756145510412,
                                                              10.536748437314278348,
                                                              10.549565591871418846,
                                                              10.562305176989406519,
                                                              10.574968132386613817,
                                                              10.587555380748748135,
                                                              10.60006782813989723,
                                                              10.612506364399679981,
                                                              10.624871863529332217,
                                                              10.637165184065759505,
                                                              10.649387169446338675,
                                                              10.661538648359456971,
                                                              10.673620435089258507,
                                                              10.685633329846911721,
                                                              10.697578119094702132,
                                                              10.709455575858433107,
                                                              10.721266460033740842,
                                                              10.733011518681895069,
                                                              10.744691486318380669,
                                                              10.756307085191693318,
                                                              10.76785902555625718,
                                                              10.77934800593665976,
                                                              10.790774713386179329,
                                                              10.802139823735060986,
                                                              10.813444001837114072,
                                                              10.82468790180285545,
                                                              10.8358721672335232,
                                                              10.846997431443936577,
                                                              10.858064317680760524,
                                                              10.869073439335851461,
                                                              10.880025400153099824,
                                                              10.890920794430101637,
                                                              10.901760207214730158,
                                                              10.912544214496701755,
                                                              10.923273383392475111,
                                                              10.933948272329693907,
                                                              10.944569431221378153,
                                                              10.955137401639886363,
                                                              10.965652716985310633,
                                                              10.976115902648308165,
                                                              10.986527476171035644,
                                                              10.996887947402781194,
                                                              11.00719781865054081,
                                                              11.017457584830406603,
                                                              11.027667733608263134,
                                                              11.037828745544826958,
                                                              11.047941094229994263,
                                                              11.058005246419710943,
                                                              11.06802166216550809,
                                                              11.077990794943804786,
                                                              11.087913091779419972,
                                                              11.097788993368448374,
                                                              11.107618934197896365,
                                                              11.117403342660917787,
                                                              11.127142641170605586,
                                                              11.136837246271941382,
                                                              11.146487568749375185,
                                                              11.156094013733063974,
                                                              11.165656980801944798,
                                                              11.175176864084914286,
                                                              11.184654052361224785,
                                                              11.194088929154311529,
                                                              11.203481872828279009,
                                                              11.212833256679813232,
                                                              11.22214344902846328,
                                                              11.231412813303665033,
                                                              11.240641708133454202,
                                                              11.24983048742707048,
                                                              11.258979500456714362,
                                                              11.268089091941565272,
                                                              11.277159602123761672,
                                                              11.286191366846269446,
                                                              11.295184717630176507,
                                                              11.304139981746262222,
                                                              11.31305748228920649,
                                                              11.321937538248453947,
                                                              11.33078046457609922,
                                                              11.339586572256690644,
                                                              11.348356168371918073,
                                                              11.357089556167919042,
                                                              11.36578703511664834,
                                                              11.374448900980526744,
                                                              11.383075445871300602,
                                                              11.39166695831309184,
                                                              11.400223723296788236,
                                                              11.408746022341706094,
                                                              11.417234133550113739,
                                                              11.425688331660353469,
                                                              11.434108888105869895,
                                                              11.442496071063368746,
                                                              11.45085014550727287,
                                                              11.45917137325921864,
                                                              11.46746001304017426,
                                                              11.475716320516829327,
                                                              11.483940548350432209,
                                                              11.492132946244762337,
                                                              11.500293760990789593,
                                                              11.508423236512726362,
                                                              11.516521613911566035,
                                                              11.52458913150951858,
                                                              11.532626024892328687,
                                                              11.540632526950561854,
                                                              11.548608867921116072,
                                                              11.55655527542666583,
                                                              11.564471974517864794,
                                                              11.572359187706947381,
                                                              11.580217135011519503,
                                                              11.588046033987483341,
                                                              11.595846099768325033,
                                                              11.60361754509874288,
                                                              11.611360580372707574,
                                                              11.619075413665630236,
                                                              11.626762250768615914,
                                                              11.634421295224266046,
                                                              11.642052748355579794,
                                                              11.649656809300974913,
                                                              11.657233675045707955,
                                                              11.664783540450647692,
                                                              11.672306598285697277,
                                                              11.67980303925796548,
                                                              11.687273052039778065,
                                                              11.694716823301339659,
                                                              11.702134537735622288,
                                                              11.709526378087792509,
                                                              11.716892525182384333,
                                                              11.724233157950031625,
                                                              11.731548453452740333,
                                                              11.738838586912624606,
                                                              11.74610373173514688,
                                                              11.753344059533525012,
                                                              11.76055974015594785,
                                                              11.767750941708468915,
                                                              11.7749178305778468,
                                                              11.782060571456211662,
                                                              11.78917932736304941,
                                                              11.796274259669399243,
                                                              11.803345528118841301,
                                                              11.810393290847994052,
                                                              11.817417704411964152,
                                                              11.824418923801028214,
                                                              11.831397102464613624,
                                                              11.838352392329960949,
                                                              11.845284943822530721,
                                                              11.85219490588682234,
                                                              11.859082426003874744,
                                                              11.865947650212342879,
                                                              11.872790723126504631,
                                                              11.879611787952603308,
                                                              11.886410986512389698,
                                                              11.893188459255847889,
                                                              11.899944345281237901,
                                                              11.906678782350832435,
                                                              11.913391906909305717,
                                                              11.920083854102388798,
                                                              11.926754757788433636,
                                                              11.933404750558930019,
                                                              11.940033963752854973,
                                                              11.94664252747291755,
                                                              11.953230570598858407,
                                                              11.95979822080824917,
                                                              11.966345604583398909,
                                                              11.972872847234746985,
                                                              11.979380072908185184,
                                                              11.985867404604579889,
                                                              11.992334964189772961,
                                                              11.998782872412620648,
                                                              12.005211248915145461,
                                                              12.011620212247880346,
                                                              12.018009879882104229,
                                                              12.024380368224580451,
                                                              12.030731792627817001,
                                                              12.037064267405851226,
                                                              12.04337790584380663,
                                                              12.049672820211730695,
                                                              12.05594912177746636,
                                                              12.062206920817002853,
                                                              12.068446326625002385,
                                                              12.074667447532235087,
                                                              12.080870390909414169,
                                                              12.087055263183344778,
                                                              12.093222169846605141,
                                                              12.099371215468432084,
                                                              12.105502503705665163,
                                                              12.111616137313081509,
                                                              12.117712218154043313,
                                                              12.123790847211116883,
                                                              12.129852124595206675,
                                                              12.135896149556476331,
                                                              12.141923020494703067,
                                                              12.147932834966560733,
                                                              12.153925689698381163,
                                                              12.159901680593627304,
                                                              12.165860902742954508,
                                                              12.171803450432264526,
                                                              12.177729417153157598,
                                                              12.183638895611851538,
                                                              12.189531977736416835,
                                                              12.195408754686605235,
                                                              12.201269316862454417,
                                                              12.207113753913702681,
                                                              12.212942154744636625,
                                                              12.218754607526607359,
                                                              12.224551199702874626,
                                                              12.230332017998598815,
                                                              12.236097148426086534,
                                                              12.241846676297585716,
                                                              12.247580686227967917,
                                                              12.25329926214290488,
                                                              12.259002487289937022,
                                                              12.264690444241271194,
                                                              12.27036321490379045,
                                                              12.27602088052585394,
                                                              12.281663521704766495,
                                                              12.287291218391240832,
                                                              12.292904049898977448,
                                                              12.298502094910508831,
                                                              12.304085431483887447,
                                                              12.309654137059869328,
                                                              12.315208288465633757,
                                                              12.32074796192694599,
                                                              12.326273233067174928,
                                                              12.331784176921180674,
                                                              12.337280867932953754,
                                                              12.342763379971831483,
                                                              12.348231786328973669,
                                                              12.353686159729406313,
                                                              12.359126572335103589,
                                                              12.36455309575179129,
                                                              12.369965801035290198,
                                                              12.375364758694480827,
                                                              12.380750038700686133,
                                                              12.386121710490597181,
                                                              12.391479842970930747,
                                                              12.396824504527426569,
                                                              12.402155763026529556,
                                                              12.407473685821198472,
                                                              12.412778339759659829,
                                                              12.418069791183780737,
                                                              12.423348105941609987,
                                                              12.428613349386717246,
                                                              12.43386558638595929,
                                                              12.439104881321272345,
                                                              12.444331298100673067,
                                                              12.449544900156126914,
                                                              12.454745750450241459,
                                                              12.45993391148516416,
                                                              12.465109445300381452,
                                                              12.470272413482664575,
                                                              12.47542287716634668,
                                                              12.480560897041524271,
                                                              12.485686533355790928,
                                                              12.490799845918079569,
                                                              12.495900894106803491,
                                                              12.500989736870524283,
                                                              12.50606643273041918,
                                                              12.511131039789567865,
                                                              12.516183615733602608,
                                                              12.521224217836033787,
                                                              12.526252902958219693,
                                                              12.531269727560710336,
                                                              12.536274747701293464,
                                                              12.541268019040277437,
                                                              12.546249596842478979,
                                                              12.55121953598744966,
                                                              12.556177890963562405,
                                                              12.561124715879925517,
                                                              12.56606006446608248,
                                                              12.570983990074571679,
                                                              12.575896545686061856,
                                                              12.580797783914913879,
                                                              12.585687757009413446,
                                                              12.590566516853797907,
                                                              12.595434114976942652,
                                                              12.600290602551172725,
                                                              12.605136030397488867,
                                                              12.60997044898745223,
                                                              12.614793908448151072,
                                                              12.619606458563602303,
                                                              12.62440814877846762,
                                                              12.629199028202942046,
                                                              12.633979145612149964,
                                                              12.638748549452740733,
                                                              12.643507287842705722,
                                                              12.648255408576973835,
                                                              12.652992959129829131,
                                                              12.657719986655116884,
                                                              12.662436537993787766,
                                                              12.667142659672373739,
                                                              12.67183839790736144,
                                                              12.676523798608469562,
                                                              12.681198907382585261,
                                                              12.685863769531220413,
                                                              12.690518430060949484,
                                                              12.695162933678602712,
                                                              12.699797324799542153,
                                                              12.704421647545103724,
                                                              12.709035945750736474,
                                                              12.713640262964121419,
                                                              12.718234642449314009,
                                                              12.722819127189644917,
                                                              12.727393759888265379,
                                                              12.731958582974320038,
                                                              12.736513638600756693,
                                                              12.741058968649468852,
                                                              12.745594614732066674,
                                                              12.750120618196341127,
                                                              12.75463702012170053,
                                                              12.75914386132594025,
                                                              12.763641182368505866,
                                                              12.768129023548373979,
                                                              12.772607424908949625,
                                                              12.777076426242249596,
                                                              12.781536067086181063,
                                                              12.785986386729678799,
                                                              12.790427424216378682,
                                                              12.794859218341448681,
                                                              12.799281807657878929,
                                                              12.80369523047976088,
                                                              12.808099524879954956,
                                                              12.812494728692886525,
                                                              12.816880879520020642,
                                                              12.821258014729650654,
                                                              12.825626171456725899,
                                                              12.829985386608969478,
                                                              12.834335696864167531,
                                                              12.838677138677613954,
                                                              12.843009748276069004,
                                                              12.847333561669227464,
                                                              12.851648614644567203,
                                                              12.855954942770035032,
                                                              12.86025258139869365,
                                                              12.864541565667426681,
                                                              12.868821930501345818,
                                                              12.873093710613842333,
                                                              12.87735694050643076,
                                                              12.881611654475847217,
                                                              12.885857886611180589,
                                                              12.890095670795693295,
                                                              12.894325040711107633,
                                                              12.898546029837236304,
                                                              12.90275867145457056,
                                                              12.906962998643509266,
                                                              12.911159044289716391,
                                                              12.915346841084431873,
                                                              12.919526421523887194,
                                                              12.923697817910088759,
                                                              12.927861062359458089,
                                                              12.93201618679557896,
                                                              12.936163222956782448,
                                                              12.940302202393013431,
                                                              12.944433156472694435,
                                                              12.948556116376956027,
                                                              12.95267111310943875,
                                                              12.956778177490662074,
                                                              12.96087734016371229,
                                                              12.964968631592270754,
                                                              12.969052082066681919,
                                                              12.973127721698544335,
                                                              12.977195580430983313,
                                                              12.981255688030060469,
                                                              12.985308074093628861,
                                                              12.989352768049899467,
                                                              12.993389799157773368,
                                                              12.997419196511796002,
                                                              13.00144098903680856,
                                                              13.005455205494589777,
                                                              13.009461874487961097,
                                                              13.013461024451787651,
                                                              13.01745268366328645,
                                                              13.021436880240200296,
                                                              13.02541364213947972,
                                                              13.029382997165184221,
                                                              13.03334497296239114,
                                                              13.03729959702083363,
                                                              13.041246896678583056,
                                                              13.045186899121286928,
                                                              13.04911963138158626,
                                                              13.053045120343410801,
                                                              13.056963392740687624,
                                                              13.060874475158223973,
                                                              13.064778394037061204,
                                                              13.068675175669824284,
                                                              13.07256484620359771,
                                                              13.076447431646739616,
                                                              13.080322957858628641,
                                                              13.084191450560513914,
                                                              13.08805293533270131,
                                                              13.091907437615986964,
                                                              13.095754982711513392,
                                                              13.09959559578430266,
                                                              13.103429301860911593,
                                                              13.107256125835576199,
                                                              13.111076092464362119,
                                                              13.114889226372408615,
                                                              13.118695552051653053,
                                                              13.122495093859603443,
                                                              13.126287876027134871,
                                                              13.130073922651970264,
                                                              13.133853257706881834,
                                                              13.137625905032018991,
                                                              13.141391888343568084,
                                                              13.145151231231277933,
                                                              13.148903957157155986,
                                                              13.152650089461509708,
                                                              13.156389651362033533,
                                                              13.160122665949137044,
                                                              13.163849156196825163,
                                                              13.167569144954404337,
                                                              13.171282654952305435,
                                                              13.174989708803149568,
                                                              13.178690328999275394,
                                                              13.182384537916657763,
                                                              13.186072357813859668,
                                                              13.189753810832621994,
                                                              13.193428919001977562,
                                                              13.197097704233886617,
                                                              13.200760188329571321,
                                                              13.204416392974891892,
                                                              13.20806633974491362,
                                                              13.211710050105512693,
                                                              13.215347545408052454,
                                                              13.218978846896556334,
                                                              13.22260397570619439,
                                                              13.226222952863775362,
                                                              13.229835799288590437,
                                                              13.233442535791176908,
                                                              13.237043183081210529,
                                                              13.240637761757254154,
                                                              13.244226292317046401,
                                                              13.247808795152753447,
                                                              13.251385290554667407,
                                                              13.254955798708774495,
                                                              13.258520339702053903,
                                                              13.262078933518273161,
                                                              13.265631600040839189,
                                                              13.269178359054219385,
                                                              13.272719230242602251,
                                                              13.276254233192984699,
                                                              13.279783387396898675,
                                                              13.283306712241415681,
                                                              13.286824227026306389,
                                                              13.290335950949151211,
                                                              13.293841903113740699,
                                                              13.297342102530555152,
                                                              13.300836568116132241,
                                                              13.304325318692056257,
                                                              13.307808372989974544,
                                                              13.311285749644119036,
                                                              13.3147574672044815,
                                                              13.318223544124418112,
                                                              13.321683998769101365,
                                                              13.32513884941529092,
                                                              13.32858811424843104,
                                                              13.332031811365043339,
                                                              13.335469958777091293,
                                                              13.33890257440550009,
                                                              13.342329676086286838,
                                                              13.345751281569015134,
                                                              13.349167408516496636,
                                                              13.352578074507405859,
                                                              13.355983297036111423,
                                                              13.359383093511311813,
                                                              13.362777481260028623,
                                                              13.366166477523949041,
                                                              13.369550099465973503,
                                                              13.372928364164186732,
                                                              13.37630128861458978,
                                                              13.379668889734126935,
                                                              13.383031184360207888,
                                                              13.386388189246655855,
                                                              13.389739921073138262,
                                                              13.393086396434743079,
                                                              13.396427631850842843,
                                                              13.399763643766091192,
                                                              13.403094448541603256,
                                                              13.406420062466038345,
                                                              13.409740501751686637,
                                                              13.413055782530316051,
                                                              13.416365920863402295,
                                                              13.419670932734781843,
                                                              13.422970834054583023,
                                                              13.426265640657085498,
                                                              13.429555368307228846,
                                                              13.432840032691371945,
                                                              13.436119649428079015,
                                                              13.439394234059173883,
                                                              13.44266380205828959,
                                                              13.445928368826187693,
                                                              13.449187949693879318,
                                                              13.45244255991741511,
                                                              13.455692214689392472,
                                                              13.458936929129038163,
                                                              13.462176718283481947,
                                                              13.465411597140072075,
                                                              13.468641580606668384,
                                                              13.47186668353329253,
                                                              13.475086920694291592,
                                                              13.478302306802110166,
                                                              13.48151285650034481,
                                                              13.484718584366349958,
                                                              13.487919504913703506,
                                                              13.491115632585090722,
                                                              13.494306981764220055,
                                                              13.497493566767404971,
                                                              13.500675401844988599,
                                                              13.503852501186502266,
                                                              13.507024878915773414,
                                                              13.510192549093989811,
                                                              13.513355525719184413,
                                                              13.5165138227280206,
                                                              13.519667453995577233,
                                                              13.522816433330204333,
                                                              13.525960774487495897,
                                                              13.529100491153101871,
                                                              13.532235596958095414,
                                                              13.535366105472615317,
                                                              13.538492030203547856,
                                                              13.541613384601010495,
                                                              13.544730182054616208,
                                                              13.54784243589776338,
                                                              13.550950159401967454,
                                                              13.554053365781573603,
                                                              13.557152068194858074,
                                                              13.560246279740981734,
                                                              13.563336013459098339,
                                                              13.566421282338698262,
                                                              13.569502099306779641,
                                                              13.572578477234690908,
                                                              13.575650428942710235,
                                                              13.578717967190852889,
                                                              13.581781104683333439,
                                                              13.584839854072622956,
                                                              13.587894227958557281,
                                                              13.590944238877442629,
                                                              13.593989899322760806,
                                                              13.597031221726449957,
                                                              13.600068218471934145,
                                                              13.603100901887216878,
                                                              13.606129284248572375,
                                                              13.609153377777310823,
                                                              13.612173194646384644,
                                                              13.615188746974233425,
                                                              13.618200046832647843,
                                                              13.621207106232205319,
                                                              13.624209937141896987,
                                                              13.627208551476659437,
                                                              13.630202961101861803,
                                                              13.633193177830367659,
                                                              13.636179213427501722,
                                                              13.639161079608454585,
                                                              13.642138788041277664,
                                                              13.645112350340662388,
                                                              13.648081778073958503,
                                                              13.651047082764348417,
                                                              13.654008275881622581,
                                                              13.656965368849332876,
                                                              13.65991837304498091,
                                                              13.662867299796056741,
                                                              13.665812160384957608,
                                                              13.668752966047126307,
                                                              13.67168972797138693,
                                                              13.674622457297903821,
                                                              13.677551165123905008,
                                                              13.680475862497145201,
                                                              13.683396560426253075,
                                                              13.686313269868923825,
                                                              13.68922600173787707,
                                                              13.69213476690431186,
                                                              13.695039576192915121,
                                                              13.697940440384053673,
                                                              13.700837370214280497,
                                                              13.703730376375016675,
                                                              13.706619469516800436,
                                                              13.70950466024423342,
                                                              13.712385959120915402,
                                                              13.715263376664893258,
                                                              13.718136923354830259,
                                                              13.721006609623605854,
                                                              13.723872445865787029,
                                                              13.72673444243013563,
                                                              13.729592609625129285,
                                                              13.732446957718616787,
                                                              13.735297496935510608,
                                                              13.738144237461392905,
                                                              13.740987189439143989,
                                                              13.743826362973667443,
                                                              13.746661768126049452,
                                                              13.749493414917923317,
                                                              13.752321313335878372,
                                                              13.755145473319212002,
                                                              13.757965904770790289,
                                                              13.76078261755586496,
                                                              13.763595621499426613,
                                                              13.766404926387352248,
                                                              13.769210541965978933,
                                                              13.772012477943038178,
                                                              13.774810743989194251,
                                                              13.777605349738051288,
                                                              13.780396304779271688,
                                                              13.783183618675410642,
                                                              13.78596730094170475,
                                                              13.788747361057883012,
                                                              13.791523808472934576,
                                                              13.794296652588764474,
                                                              13.797065902780534685,
                                                              13.799831568380581004,
                                                              13.802593658685578859,
                                                              13.805352182958312568,
                                                              13.80810715042531811,
                                                              13.81085857027452235,
                                                              13.813606451660854546,
                                                              13.816350803700302663,
                                                              13.819091635480996061,
                                                              13.821828956047957249,
                                                              13.824562774414131283,
                                                              13.827293099560392875,
                                                              13.83001994042906091,
                                                              13.832743305931924027,
                                                              13.835463204941126136,
                                                              13.838179646300201142,
                                                              13.840892638819539684,
                                                              13.843602191269319235,
                                                              13.846308312392787698,
                                                              13.84901101089392661,
                                                              13.851710295449825239,
                                                              13.854406174701319188,
                                                              13.857098657254422136,
                                                              13.859787751686601709,
                                                              13.862473466541130662,
                                                              13.86515581032993083,
                                                              13.867834791527137384,
                                                              13.870510418584625967,
                                                              13.873182699914616833,
                                                              13.875851643901116361,
                                                              13.878517258893786845,
                                                              13.881179553217378952,
                                                              13.883838535155623717,
                                                              13.886494212968770512,
                                                              13.889146594887002095,
                                                              13.891795689101494204,
                                                              13.894441503781525427,
                                                              13.897084047059651724,
                                                              13.899723327045110821,
                                                              13.902359351811346855,
                                                              13.904992129401231082,
                                                              13.907621667830589729,
                                                              13.910247975086692307,
                                                              13.912871059123370188,
                                                              13.915490927870026283,
                                                              13.91810758922001412,
                                                              13.920721051044457894,
                                                              13.923331321182732978,
                                                              13.925938407443748446,
                                                              13.928542317608103573,
                                                              13.931143059431320808,
                                                              13.933740640637571673,
                                                              13.936335068925462366,
                                                              13.938926351958125238,
                                                              13.941514497381966464,
                                                              13.944099512804314145,
                                                              13.94668140581365634,
                                                              13.949260183964351967,
                                                              13.951835854788662061,
                                                              13.954408425790269632,
                                                              13.95697790443957409,
                                                              13.959544298190312617,
                                                              13.962107614460476768,
                                                              13.96466786064560317,
                                                              13.967225044112664634,
                                                              13.969779172205697648,
                                                              13.972330252237139092,
                                                              13.97487829149502403,
                                                              13.977423297246341249,
                                                              13.979965276723488543,
                                                              13.982504237137508341,
                                                              13.985040185675003954,
                                                              13.987573129492263391,
                                                              13.990103075725903281,
                                                              13.992630031483724196,
                                                              13.99515400384691155,
                                                              13.997674999873927604,
                                                              14.000193026599195179,
                                                              14.002708091029330006,
                                                              14.005220200146053955,
                                                              14.007729360909634053,
                                                              14.01023558025232596,
                                                              14.01273886508432831,
                                                              14.01523922228879826,
                                                              14.017736658729967658,
                                                              14.020231181239813267,
                                                              14.022722796634397469,
                                                              14.025211511700168998,
                                                              14.027697333202896246,
                                                              14.030180267882093048,
                                                              14.03266032245799444,
                                                              14.035137503624289579,
                                                              14.037611818050047319,
                                                              14.040083272382236856,
                                                              14.04255187324901577,
                                                              14.045017627248094882,
                                                              14.04748054095911769,
                                                              14.04994062093845919,
                                                              14.052397873716437005,
                                                              14.054852305807788326,
                                                              14.057303923696416348,
                                                              14.059752733849744999,
                                                              14.062198742710839028,
                                                              14.0646419567001697,
                                                              14.067082382216762326,
                                                              14.069520025637352489,
                                                              14.071954893316462432,
                                                              14.07438699158885953,
                                                              14.076816326763628595,
                                                              14.07924290513405019,
                                                              14.081666732966468203,
                                                              14.084087816508418456,
                                                              14.086506161985690611,
                                                              14.088921775603690634,
                                                              14.091334663544339278,
                                                              14.093744831971463327,
                                                              14.096152287025153882,
                                                              14.09855703482939937,
                                                              14.100959081480601398,
                                                              14.103358433059241861,
                                                              14.105755095624877171,
                                                              14.10814907521682926,
                                                              14.110540377851714666,
                                                              14.112929009528890845,
                                                              14.115314976224503596,
                                                              14.117698283896190858,
                                                              14.120078938482535591,
                                                              14.122456945901159742,
                                                              14.124832312049791838,
                                                              14.127205042808224533,
                                                              14.129575144031136347,
                                                              14.131942621562323836,
                                                              14.13430748121819569,
                                                              14.136669728799523327,
                                                              14.139029370088032067,
                                                              14.141386410844752675,
                                                              14.143740856811602313,
                                                              14.146092713715217926,
                                                              14.148441987258625119,
                                                              14.1507886831258336,
                                                              14.153132806984910275,
                                                              14.155474364484195604,
                                                              14.157813361255202622,
                                                              14.160149802908009065,
                                                              14.162483695034042697,
                                                              14.164815043210477796,
                                                              14.167143852989557473,
                                                              14.169470129910484602,
                                                              14.171793879496814128,
                                                              14.17411510724339152,
                                                              14.176433818640294504,
                                                              14.178750019150349715,
                                                              14.181063714220734084,
                                                              14.183374909285145549,
                                                              14.185683609751258416,
                                                              14.187989821018593517,
                                                              14.19029354846252744,
                                                              14.192594797443804921,
                                                              14.194893573304014467,
                                                              14.197189881369276065,
                                                              14.199483726948839646,
                                                              14.20177511533286463,
                                                              14.204064051795191048,
                                                              14.206350541593439019,
                                                              14.208634589965885908,
                                                              14.210916202138053066,
                                                              14.213195383316227449,
                                                              14.215472138690623538,
                                                              14.21774647343459641,
                                                              14.220018392702060694,
                                                              14.222287901634690144,
                                                              14.224555005357652604,
                                                              14.226819708976794487,
                                                              14.229082017582555864,
                                                              14.231341936254585434,
                                                              14.233599470045575686,
                                                              14.235854624000007007,
                                                              14.238107403148534402,
                                                              14.240357812495970435,
                                                              14.24260585704149662,
                                                              14.244851541762988845,
                                                              14.247094871621490242,
                                                              14.249335851570357647,
                                                              14.251574486535515618,
                                                              14.253810781437111999,
                                                              14.256044741176957302,
                                                              14.258276370639412889,
                                                              14.260505674693048306,
                                                              14.262732658195368174,
                                                              14.264957325986895142,
                                                              14.267179682890123615,
                                                              14.269399733716818801,
                                                              14.271617483259406711,
                                                              14.273832936298923357,
                                                              14.276046097600476159,
                                                              14.278256971912142603,
                                                              14.280465563973503862,
                                                              14.282671878500293516,
                                                              14.284875920200541088,
                                                              14.287077693764764774,
                                                              14.289277203869636068,
                                                              14.291474455178866165,
                                                              14.293669452337670833,
                                                              14.29586219998199681,
                                                              14.29805270272855644,
                                                              14.300240965185764708,
                                                              14.302426991941993961,
                                                              14.304610787573965425,
                                                              14.306792356644967512,
                                                              14.308971703701400813,
                                                              14.311148833281771786,
                                                              14.313323749904027338,
                                                              14.315496458076189867,
                                                              14.317666962290227062,
                                                              14.319835267025963432,
                                                              14.322001376753128454,
                                                              14.324165295916989393,
                                                              14.326327028962005272,
                                                              14.328486580308904763,
                                                              14.330643954371728555,
                                                              14.33279915554975581,
                                                              14.334952188226184333,
                                                              14.337103056773381482,
                                                              14.339251765549725803,
                                                              14.341398318899807762,
                                                              14.343542721156270048,
                                                              14.345684976638173325,
                                                              14.34782508965116854,
                                                              14.349963064492426312,
                                                              14.35209890543410971,
                                                              14.354232616751593454,
                                                              14.356364202695091947,
                                                              14.358493667506255775,
                                                              14.360621015414713142,
                                                              14.362746250639503387,
                                                              14.364869377382294857,
                                                              14.366990399835495751,
                                                              14.36910932217366188},
                                                             {0,
                                                              2.3219280948873626258,
                                                              3.3219280948873626258,
                                                              4.009984184985102118,
                                                              4.5410966501985230437,
                                                              4.9759235872791061084,
                                                              5.3451175544612743451,
                                                              5.6664899140623274931,
                                                              5.951365559463460464,
                                                              6.2074181133603802962,
                                                              6.4400996405551707014,
                                                              6.6534295905427747186,
                                                              6.8504593992211040288,
                                                              7.0335637469580500891,
                                                              7.2046269409945082884,
                                                              7.3651708911261204449,
                                                              7.5164434474353214455,
                                                              7.65948003120419596,
                                                              7.7951500049725712671,
                                                              7.924190846958517831,
                                                              8.0472329877362867023,
                                                              8.164820486503929331,
                                                              8.2774261368204520295,
                                                              8.3854626148977526867,
                                                              8.4892931373039921539,
                                                              8.5892394441002384298,
                                                              8.6855861004875691833,
                                                              8.7785887576787668962,
                                                              8.8684753364507304951,
                                                              8.9554515670237062608,
                                                              9.0397030433341374334,
                                                              9.1213986619859461769,
                                                              9.2006909239974170589,
                                                              9.2777196513117612398,
                                                              9.3526126611213786788,
                                                              9.4254869321160441586,
                                                              9.4964501859713710274,
                                                              9.5656015283077344691,
                                                              9.6330329117983168885,
                                                              9.6988286850167657605,
                                                              9.763067719441490766,
                                                              9.8258230457748947373,
                                                              9.8871625158589804272,
                                                              9.9471498081558671345,
                                                              10.005843953918361677,
                                                              10.063300157904480514,
                                                              10.119570450540107842,
                                                              10.174703458716329862,
                                                              10.228744679631718739,
                                                              10.281736906750806426,
                                                              10.333721504103394295,
                                                              10.384736055219464745,
                                                              10.434816581828172843,
                                                              10.483997118677041982,
                                                              10.532309860431933402,
                                                              10.579785293150695935,
                                                              10.626452313782410286,
                                                              10.672338339249707317,
                                                              10.717469406258699038,
                                                              10.761870262792360364,
                                                              10.805564452116826146,
                                                              10.848574390029094516,
                                                              10.890921435992128252,
                                                              10.932625958730842086,
                                                              10.973707396799733615,
                                                              11.014184314579676283,
                                                              11.05407445411208478,
                                                              11.093394783137521031,
                                                              11.132161539667729144,
                                                              11.17039027338819146,
                                                              11.20809588415779956,
                                                              11.24529265784756582,
                                                              11.281994299735856657,
                                                              11.318213965658326714,
                                                              11.35396429109132832,
                                                              11.389257418331439453,
                                                              11.424105021919139347,
                                                              11.458518332441173015,
                                                              11.492508158834391807,
                                                              11.526084909303103032,
                                                              11.559258610951991031,
                                                              11.592038928228676653,
                                                              11.624435180261382783,
                                                              11.65645635717043227,
                                                              11.688111135425632625,
                                                              11.719407892315885533,
                                                              11.750354719592182917,
                                                              11.780959436339754731,
                                                              11.811229601131572053,
                                                              11.841172523510511638,
                                                              11.870795274844477163,
                                                              11.900104698595182384,
                                                              11.929107420038091547,
                                                              11.957809855468735094,
                                                              11.986218220927369771,
                                                              12.014338540471927175,
                                                              12.042176654027532123,
                                                              12.06973822483764458,
                                                              12.097028746541726107,
                                                              12.124053549900946081,
                                                              12.150817809193112495,
                                                              12.177326548296401043,
                                                              12.203584646479432863,
                                                              12.229596843914722015,
                                                              12.255367746931701944,
                                                              12.280901833023179393,
                                                              12.306203455619561637,
                                                              12.331276848643559774,
                                                              12.356126130857282064,
                                                              12.380755310013151771,
                                                              12.405168286818895496,
                                                              12.429368858727011471,
                                                              12.453360723557475254,
                                                              12.477147482962628544,
                                                              12.500732645742624882,
                                                              12.524119631018825416,
                                                              12.547311771272545045,
                                                              12.570312315255755209,
                                                              12.593124430780569867,
                                                              12.615751207393291367,
                                                              12.63819565893856911,
                                                              12.660460726019156397,
                                                              12.682549278356763267,
                                                              12.704464117057906236,
                                                              12.72620797678986726,
                                                              12.747783527871257903,
                                                              12.769193378280535356,
                                                              12.790440075586433366,
                                                              12.811526108804656587,
                                                              12.832453910183094337,
                                                              12.853225856919832992,
                                                              12.873844272816173273,
                                                              12.89431142986833656,
                                                              12.914629549799643726,
                                                              12.93480080553720768,
                                                              12.954827322633622799,
                                                              12.974711180638211161,
                                                              12.994454414418219912,
                                                              13.014059015433037558,
                                                              13.033526932963361844,
                                                              13.052860075297083142,
                                                              13.07206031087321918,
                                                              13.091129469386942219,
                                                              13.110069342856098373,
                                                              13.12888168665087818,
                                                              13.14756822048918572,
                                                              13.166130629397750695,
                                                              13.184570564641676427,
                                                              13.202889644622032961,
                                                              13.221089455744994723,
                                                              13.239171553261353864,
                                                              13.257137462079056078,
                                                              13.274988677549504956,
                                                              13.292726666228245946,
                                                              13.310352866611388833,
                                                              13.327868689848640926,
                                                              13.345275520433842686,
                                                              13.362574716873641734,
                                                              13.379767612335315974,
                                                              13.396855515274722848,
                                                              13.413839710044697995,
                                                              13.430721457484715131,
                                                              13.447501995492977755,
                                                              13.464182539580649589,
                                                              13.480764283409936155,
                                                              13.497248399315717293,
                                                              13.513636038811856821,
                                                              13.529928333082400727,
                                                              13.546126393458756354,
                                                              13.562231311882127827,
                                                              13.578244161353515196,
                                                              13.59416599636993972,
                                                              13.609997853348680508,
                                                              13.625740751038870613,
                                                              13.641395690922031747,
                                                              13.65696365760090103,
                                                              13.672445619177601372,
                                                              13.687842527620885491,
                                                              13.703155319123602851,
                                                              13.718384914450053813,
                                                              13.733532219274177777,
                                                              13.748598124507793727,
                                                              13.763583506621067443,
                                                              13.778489227952839258,
                                                              13.793316137014183909,
                                                              13.808065068782807927,
                                                              13.822736844989913507,
                                                              13.837332274399686938,
                                                              13.851852153081235741,
                                                              13.866297264673491441,
                                                              13.880668380643250259,
                                                              13.894966260536559588,
                                                              13.909191652223285018,
                                                              13.923345292136307449,
                                                              13.937427905503477987,
                                                              13.951440206574563518,
                                                              13.965382898842245041,
                                                              13.979256675257280662,
                                                              13.993062218438765854,
                                                              14.006800200878895879,
                                                              14.020471285142434681,
                                                              14.034076124062194069,
                                                              14.047615360928320527,
                                                              14.06108962967455156,
                                                              14.074499555058505251,
                                                              14.087845752838797253,
                                                              14.10112882994708805,
                                                              14.114349384656605935,
                                                              14.127508006746113622,
                                                              14.140605277660474925,
                                                              14.153641770667354294,
                                                              14.166618051009944423,
                                                              14.179534676056375631,
                                                              14.192392195445822978,
                                                              14.205191151230881275,
                                                              14.217932078016900732,
                                                              14.230615503097977737,
                                                              14.243241946589959568,
                                                              14.255811921560713529,
                                                              14.26832593415661421,
                                                              14.280784483727142486,
                                                              14.293188062946203587,
                                                              14.305537157930841019,
                                                              14.317832248357067471,
                                                              14.330073807573846523,
                                                              14.342262302713766786,
                                                              14.354398194801470723,
                                                              14.366481938860525602,
                                                              14.37851398401674885,
                                                              14.390494773600016387,
                                                              14.402424745244067239,
                                                              14.414304330983403801,
                                                              14.426133957348937997,
                                                              14.437914045461239354,
                                                              14.449645011121564409,
                                                              14.461327264901584044,
                                                              14.472961212230551098,
                                                              14.484547253481402151,
                                                              14.496085784054198342,
                                                              14.507577194458724179,
                                                              14.519021870394517748,
                                                              14.53042019283030406,
                                                              14.541772538080387278,
                                                              14.553079277881092679,
                                                              14.564340779464544084,
                                                              14.575557405630885199,
                                                              14.586729514820371278,
                                                              14.597857461182197625,
                                                              14.608941594643322759,
                                                              14.619982260975177724,
                                                              14.63097980185975544,
                                                              14.641934554953504488,
                                                              14.652846853950602934,
                                                              14.663717028645002927,
                                                              14.674545404990841035,
                                                              14.685332305162079436,
                                                              14.696078047610706463,
                                                              14.7067829471241307,
                                                              14.717447314881251685,
                                                              14.728071458507404401,
                                                              14.738655682128454671,
                                                              14.74920028642363512,
                                                              14.759705568678020526,
                                                              14.770171822832590891,
                                                              14.780599339535331893,
                                                              14.790988406189555349,
                                                              14.801339307002578494,
                                                              14.811652323032616252,
                                                              14.821927732235725017,
                                                              14.832165809510927446,
                                                              14.842366826745026387,
                                                              14.852531052856953409,
                                                              14.862658753840213066,
                                                              14.872750192805749947,
                                                              14.882805630023494103,
                                                              14.892825322962799817,
                                                              14.902809526333083312,
                                                              14.912758492122891951,
                                                              14.922672469638355253,
                                                              14.932551705541857956,
                                                              14.942396443888757318,
                                                              14.952206926164333112,
                                                              14.961983391319785497,
                                                              14.971726075807604062,
                                                              14.981435213616160596,
                                                              14.991111036304362258,
                                                              15.000753773034849914,
                                                              15.010363650606890928,
                                                              15.019940893489275524,
                                                              15.029485723851959023,
                                                              15.038998361597258224,
                                                              15.048479024390861269,
                                                              15.057927927691952874,
                                                              15.067345284783257853,
                                                              15.076731306800107646,
                                                              15.086086202758982822,
                                                              15.095410179586501442,
                                                              15.104703442146483638,
                                                              15.113966193267872384,
                                                              15.123198633771341548,
                                                              15.132400962495822228,
                                                              15.141573376324718225,
                                                              15.150716070211185382,
                                                              15.159829237203657826,
                                                              15.168913068470484262,
                                                              15.177967753323947875,
                                                              15.186993479245153083,
                                                              15.195990431906743368,
                                                              15.204958795196526822,
                                                              15.213898751240062523,
                                                              15.22281048042335172,
                                                              15.231694161415287425,
                                                              15.240549971188583456,
                                                              15.249378085041929154,
                                                              15.258178676620790526,
                                                              15.266951917938254013,
                                                              15.275697979395104653,
                                                              15.284417029800843935,
                                                              15.293109236392155026,
                                                              15.30177476485382293,
                                                              15.310413779336720097,
                                                              15.319026442477570171,
                                                              15.327612915416869654,
                                                              15.336173357817893148,
                                                              15.344707927884149612,
                                                              15.353216782377456795,
                                                              15.361700076635271373,
                                                              15.370157964588184285,
                                                              15.378590598776474607,
                                                              15.386998130367325999,
                                                              15.395380709170815692,
                                                              15.403738483656383096,
                                                              15.412071600968927143,
                                                              15.420380206944292567,
                                                              15.428664446124571441,
                                                              15.436924461774330197,
                                                              15.445160395894429683,
                                                              15.453372389237655327,
                                                              15.461560581323100294,
                                                              15.469725110450553984,
                                                              15.477866113714734198,
                                                              15.48598372701921555,
                                                              15.494078085090247754,
                                                              15.502149321490390932,
                                                              15.510197568631737042,
                                                              15.518222957789250316,
                                                              15.526225619113853682,
                                                              15.534205681644861485,
                                                              15.542163273323119199,
                                                              15.55009852100315193,
                                                              15.558011550465563388,
                                                              15.565902486428969453,
                                                              15.573771452562155559,
                                                              15.581618571495846837,
                                                              15.58944396483406436,
                                                              15.597247753165763839,
                                                              15.605030056076110156,
                                                              15.61279099215777677,
                                                              15.620530679021365827,
                                                              15.628249233306904742,
                                                              15.635946770694022945,
                                                              15.643623405912785884,
                                                              15.651279252753669269,
                                                              15.658914424078590244,
                                                              15.666529031830227936,
                                                              15.674123187042097172,
                                                              15.681696999848927732,
                                                              15.689250579495666926,
                                                              15.696784034347469827,
                                                              15.704297471899081984,
                                                              15.711790998784220363,
                                                              15.719264720784371647,
                                                              15.726718742838274423,
                                                              15.734153169050728138,
                                                              15.741568102701412712,
                                                              15.748963646253692161,
                                                              15.756339901362776956,
                                                              15.763696968885094307,
                                                              15.771034948885528593,
                                                              15.778353940646855591,
                                                              15.785654042676592113,
                                                              15.792935352716584774,
                                                              15.800197967749502581,
                                                              15.807441984007605029,
                                                              15.814667496980062467,
                                                              15.821874601420745421,
                                                              15.829063391355868262,
                                                              15.8362339600910218,
                                                              15.84338640021911182,
                                                              15.85052080362721405,
                                                              15.857637261503635173,
                                                              15.864735864345528071,
                                                              15.871816701965334673,
                                                              15.878879863497770586,
                                                              15.885925437407175664,
                                                              15.892953511493228547,
                                                              15.899964172898714665,
                                                              15.906957508115239008,
                                                              15.913933602990130822,
                                                              15.920892542732380193,
                                                              15.927834411919890911,
                                                              15.934759294504681648,
                                                              15.941667273819362549,
                                                              15.948558432583851641,
                                                              15.955432852910384156,
                                                              15.962290616310255587,
                                                              15.969131803699074368,
                                                              15.975956495403114133,
                                                              15.982764771164655215,
                                                              15.989556710147573071,
                                                              15.996332390943504009,
                                                              16.003091891576833206,
                                                              16.009835289509922518,
                                                              16.016562661649235366,
                                                              16.023274084350191515,
                                                              16.029969633422599173,
                                                              16.036649384135277074,
                                                              16.043313411222285936,
                                                              16.049961788887149083,
                                                              16.056594590808071388,
                                                              16.06321189014266082,
                                                              16.069813759533747799,
                                                              16.076400271112841978,
                                                              16.082971496506125675,
                                                              16.089527506838599891,
                                                              16.096068372738621122,
                                                              16.102594164342761474,
                                                              16.109104951300661668,
                                                              16.115600802779091794,
                                                              16.122081787466211011,
                                                              16.128547973576740304,
                                                              16.134999428855678616,
                                                              16.141436220582889405,
                                                              16.147858415577157842,
                                                              16.15426608020066368,
                                                              16.160659280362882129,
                                                              16.167038081524790272,
                                                              16.173402548703187165,
                                                              16.179752746474186154,
                                                              16.186088738977709056,
                                                              16.192410589920971375,
                                                              16.198718362582724239,
                                                              16.205012119817048699,
                                                              16.211291924056759228,
                                                              16.217557837317844616,
                                                              16.223809921202544615,
                                                              16.230048236903265035,
                                                              16.236272845206283222,
                                                              16.242483806495499721,
                                                              16.248681180755276898,
                                                              16.254865027575021941,
                                                              16.261035406151624017,
                                                              16.267192375293703321,
                                                              16.273335993424268509,
                                                              16.279466318584869811,
                                                              16.285583408438316866,
                                                              16.291687320272192352,
                                                              16.29777811100215601,
                                                              16.303855837174928922,
                                                              16.309920554971963469,
                                                              16.315972320211983515,
                                                              16.322011188354480282,
                                                              16.32803721450256873,
                                                              16.334050453406625536,
                                                              16.340050959466346114,
                                                              16.346038786734354176,
                                                              16.35201398891931035,
                                                              16.357976619388217898,
                                                              16.363926731169687656,
                                                              16.369864376957092844,
                                                              16.375789609110704248,
                                                              16.381702479660933847,
                                                              16.38760304031129067,
                                                              16.393491342440427161,
                                                              16.39936743710541478,
                                                              16.405231375044614595,
                                                              16.411083206679727198,
                                                              16.416922982118538954,
                                                              16.42275075115799865,
                                                              16.428566563286491231,
                                                              16.434370467686111539,
                                                              16.440162513235755171,
                                                              16.445942748512983655,
                                                              16.451711221797250317,
                                                              16.457467981071285834,
                                                              16.463213074024810822,
                                                              16.468946548055996004,
                                                              16.474668450273952658,
                                                              16.480378827501461103,
                                                              16.486077726276892719,
                                                              16.491765192856849609,
                                                              16.497441273218242941,
                                                              16.503106013060264701,
                                                              16.508759457807375526,
                                                              16.514401652610878557,
                                                              16.520032642351136332,
                                                              16.525652471640096763,
                                                              16.531261184823158317,
                                                              16.536858825981433085,
                                                              16.542445438933519597,
                                                              16.548021067238092741,
                                                              16.553585754195804469,
                                                              16.559139542851017524,
                                                              16.56468247599375232,
                                                              16.570214596162603726,
                                                              16.575735945645583058,
                                                              16.58124656648287143,
                                                              16.586746500468152021,
                                                              16.592235789151260406,
                                                              16.597714473839104699,
                                                              16.603182595598692473,
                                                              16.608640195258043804,
                                                              16.614087313408600011,
                                                              16.619523990406634084,
                                                              16.624950266375659425,
                                                              16.630366181207346443,
                                                              16.635771774564467762,
                                                              16.641167085881356513,
                                                              16.646552154366641929,
                                                              16.651927019004610031,
                                                              16.657291718556798799,
                                                              16.662646291564161771,
                                                              16.667990776347878068,
                                                              16.67332521101173981,
                                                              16.678649633444148748,
                                                              16.683964081318414685,
                                                              16.689268592095569232,
                                                              16.694563203025555964,
                                                              16.699847951148498737,
                                                              16.7051228732971353,
                                                              16.710388006097431912,
                                                              16.715643385970452073,
                                                              16.72088904913420393,
                                                              16.726125031604901494,
                                                              16.731351369198250723,
                                                              16.736568097531343113,
                                                              16.741775252023799681,
                                                              16.746972867899227566,
                                                              16.752160980187191797,
                                                              16.757339623723915167,
                                                              16.762508833153923149,
                                                              16.767668642932196832,
                                                              16.772819087324098319,
                                                              16.777960200408021052,
                                                              16.78309201607625667,
                                                              16.788214568036401886,
                                                              16.793327889812555753,
                                                              16.798432014746957464,
                                                              16.803526976000899396,
                                                              16.808612806556670449,
                                                              16.813689539217989477,
                                                              16.818757206612087174,
                                                              16.823815841190555176,
                                                              16.828865475230337267,
                                                              16.833906140835850351,
                                                              16.838937869939254455,
                                                              16.843960694302634096,
                                                              16.848974645518083548,
                                                              16.853979755009937946,
                                                              16.858976054035494485,
                                                              16.863963573685946784,
                                                              16.868942344888196772,
                                                              16.873912398405796154,
                                                              16.878873764839386951,
                                                              16.883826474629124448,
                                                              16.88877055805463101,
                                                              16.893706045236768887,
                                                              16.89863296613876642,
                                                              16.903551350566903722,
                                                              16.908461228171997703,
                                                              16.913362628450251179,
                                                              16.918255580744297362,
                                                              16.923140114244670684,
                                                              16.928016257990233129,
                                                              16.932884040869893738,
                                                              16.937743491622981651,
                                                              16.9425946388407489,
                                                              16.947437510967478858,
                                                              16.952272136300749139,
                                                              16.957098542993239931,
                                                              16.96191675905342322,
                                                              16.966726812346532682,
                                                              16.97152873059555489,
                                                              16.976322541381929199,
                                                              16.981108272147434235,
                                                              16.98588595019374381,
                                                              16.990655602684576309,
                                                              16.995417256645946935,
                                                              17.000170938967471557,
                                                              17.004916676402888953,
                                                              17.009654495571563615,
                                                              17.014384422958759302,
                                                              17.019106484916822097,
                                                              17.023820707666160956,
                                                              17.028527117295869431,
                                                              17.033225739764919382,
                                                              17.037916600903137976,
                                                              17.042599726410895045,
                                                              17.047275141861980785,
                                                              17.051942872702550602,
                                                              17.056602944252908571,
                                                              17.061255381708321011,
                                                              17.065900210139698601,
                                                              17.070537454494196794,
                                                              17.075167139596587162,
                                                              17.079789290149054892,
                                                              17.084403930733540022,
                                                              17.089011085810952295,
                                                              17.093610779722894222,
                                                              17.09820303669236452,
                                                              17.102787880824081412,
                                                              17.107365336105484488,
                                                              17.111935426407960392,
                                                              17.116498175486722033,
                                                              17.121053606982243878,
                                                              17.125601744420752226,
                                                              17.130142611214846937,
                                                              17.13467623066448553,
                                                              17.139202625957615567,
                                                              17.143721820170679138,
                                                              17.148233836269717756,
                                                              17.152738697110592625,
                                                              17.157236425440402172,
                                                              17.16172704389739323,
                                                              17.166210575011906059,
                                                              17.170687041207379764,
                                                              17.175156464800888756,
                                                              17.179618868003288412,
                                                              17.1840742729209488,
                                                              17.188522701555097427,
                                                              17.192964175803492566,
                                                              17.197398717461133799,
                                                              17.201826348219785956,
                                                              17.206247089669904682,
                                                              17.210660963300732362,
                                                              17.215067990500390493,
                                                              17.219468192557663144,
                                                              17.223861590661691423,
                                                              17.228248205903000212,
                                                              17.232628059273807253,
                                                              17.237001171669344757,
                                                              17.241367563887312286,
                                                              17.245727256629706403,
                                                              17.250080270502291313,
                                                              17.254426626016165613,
                                                              17.258766343587716108,
                                                              17.263099443539541511,
                                                              17.267425946100338763,
                                                              17.271745871406800177,
                                                              17.276059239502956189,
                                                              17.280366070340821949,
                                                              17.284666383782386845,
                                                              17.288960199597774192,
                                                              17.293247537468293018,
                                                              17.297528416984857103,
                                                              17.301802857650077527,
                                                              17.306070878877939379,
                                                              17.31033249999460466,
                                                              17.314587740239179681,
                                                              17.318836618763349122,
                                                              17.323079154633266086,
                                                              17.327315366829022736,
                                                              17.331545274245296895,
                                                              17.335768895692414304,
                                                              17.339986249896234938,
                                                              17.344197355498977231,
                                                              17.348402231059612433,
                                                              17.352600895054518304,
                                                              17.356793365877905444,
                                                              17.360979661841657418,
                                                              17.365159801177412646,
                                                              17.369333802034987002,
                                                              17.373501682484608466,
                                                              17.377663460516604488,
                                                              17.381819154041554754,
                                                              17.38596878089148845,
                                                              17.39011235881985229,
                                                              17.394249905502434217,
                                                              17.398381438537139587,
                                                              17.402506975444918425,
                                                              17.406626533670333856,
                                                              17.41074013058159764,
                                                              17.41484778347087925,
                                                              17.418949509555904598,
                                                              17.42304532597886535,
                                                              17.427135249807612638,
                                                              17.431219298036570109,
                                                              17.435297487586080223,
                                                              17.439369835303278222,
                                                              17.443436357962969652,
                                                              17.447497072267900364,
                                                              17.451551994847879001,
                                                              17.455601142262541003,
                                                              17.459644530999561596,
                                                              17.463682177476531621,
                                                              17.467714098040527659,
                                                              17.471740308968911393,
                                                              17.475760826469311837,
                                                              17.479775666680737345,
                                                              17.483784845673248753,
                                                              17.48778837944919573,
                                                              17.491786283941920033,
                                                              17.495778575018331225,
                                                              17.499765268477847968,
                                                              17.503746380053250675,
                                                              17.507721925410677954,
                                                              17.511691920150553869,
                                                              17.515656379807619913,
                                                              17.519615319851602919,
                                                              17.523568755686863341,
                                                              17.527516702653390013,
                                                              17.531459176027503588,
                                                              17.535396191021135337,
                                                              17.539327762782946252,
                                                              17.543253906398700082,
                                                              17.547174636891110566,
                                                              17.551089969220726061,
                                                              17.55499991828603612,
                                                              17.558904498923816107,
                                                              17.56280372590921246,
                                                              17.566697613956709034,
                                                              17.570586177719679455,
                                                              17.574469431791861496,
                                                              17.57834739070633745,
                                                              17.582220068936514679,
                                                              17.586087480896782864,
                                                              17.589949640942283082,
                                                              17.593806563369430052,
                                                              17.597658262416263852,
                                                              17.601504752262602693,
                                                              17.605346047030799639,
                                                              17.609182160785461946,
                                                              17.613013107534300161,
                                                              17.616838901227882985,
                                                              17.620659555760632031,
                                                              17.624475084970370631,
                                                              17.628285502639439386,
                                                              17.632090822494131288,
                                                              17.635891058205594106,
                                                              17.639686223390050657,
                                                              17.643476331608823671,
                                                              17.647261396368989494,
                                                              17.651041431123289271,
                                                              17.654816449270906986,
                                                              17.658586464157085771,
                                                              17.662351489073873978,
                                                              17.666111537260881903,
                                                              17.669866621904166237,
                                                              17.673616756137700889,
                                                              17.677361953043504883,
                                                              17.681102225651589066,
                                                              17.684837586940343357,
                                                              17.688568049836735696,
                                                              17.692293627216667318,
                                                              17.696014331905388417,
                                                              17.699730176677736182,
                                                              17.703441174257626756,
                                                              17.707147337320158442,
                                                              17.710848678489600871,
                                                              17.714545210340972403,
                                                              17.718236945400757776,
                                                              17.721923896145383992,
                                                              17.725606075003664586,
                                                              17.729283494355279061,
                                                              17.732956166531788966,
                                                              17.736624103817053566,
                                                              17.740287318447204967,
                                                              17.7439458226106197,
                                                              17.7475996284487465,
                                                              17.751248748055697746,
                                                              17.754893193479659885,
                                                              17.758532976721472352,
                                                              17.762168109736034438,
                                                              17.76579860443253267,
                                                              17.769424472673936322,
                                                              17.773045726278112966,
                                                              17.77666237701702201,
                                                              17.780274436618149991,
                                                              17.78388191676421215,
                                                              17.78748482909260531,
                                                              17.791083185197049232,
                                                              17.794676996626467513,
                                                              17.798266274886522353,
                                                              17.801851031438815198,
                                                              17.805431277701234904,
                                                              17.809007025048853023,
                                                              17.81257828481349037,
                                                              17.816145068284257036,
                                                              17.819707386707019481,
                                                              17.823265251286301236,
                                                              17.826818673183591812,
                                                              17.830367663518767785,
                                                              17.833912233369645151,
                                                              17.837452393773155279,
                                                              17.840988155723845665,
                                                              17.844519530175912081,
                                                              17.848046528042708303,
                                                              17.851569160195964514,
                                                              17.855087437467808797,
                                                              17.858601370649363815,
                                                              17.862110970492363293,
                                                              17.865616247708004494,
                                                              17.869117212967694286,
                                                              17.872613876903582053,
                                                              17.876106250108616535,
                                                              17.879594343136158585,
                                                              17.883078166501164219,
                                                              17.886557730679083278,
                                                              17.890033046107532755,
                                                              17.893504123185451249,
                                                              17.896970972273397393,
                                                              17.900433603693965523,
                                                              17.903892027732293712,
                                                              17.907346254635566396,
                                                              17.91079629461380307,
                                                              17.91424215783896301,
                                                              17.917683854447119529,
                                                              17.921121394536235982,
                                                              17.924554788168489239,
                                                              17.927984045368805965,
                                                              17.931409176126305027,
                                                              17.934830190393768135,
                                                              17.938247098087309439,
                                                              17.941659909088286895,
                                                              17.945068633241451295,
                                                              17.948473280356495252,
                                                              17.951873860207587796,
                                                              17.955270382534095575,
                                                              17.958662857039847438,
                                                              17.962051293394043938,
                                                              17.965435701231751153,
                                                              17.968816090152635923,
                                                              17.972192469722518382,
                                                              17.975564849473183671,
                                                              17.97893323890192363,
                                                              17.982297647472794466,
                                                              17.985658084615817387,
                                                              17.989014559727348086,
                                                              17.992367082170634518,
                                                              17.995715661275820452,
                                                              17.999060306339703885,
                                                              18.002401026626170477,
                                                              18.005737831366715795,
                                                              18.009070729759883989,
                                                              18.01239973097204583,
                                                              18.015724844137011473,
                                                              18.019046078357007445,
                                                              18.022363442701447411,
                                                              18.025676946208754714,
                                                              18.028986597885083398,
                                                              18.032292406705376919,
                                                              18.035594381612927606,
                                                              18.03889253152023997,
                                                              18.042186865308398325,
                                                              18.045477391827411395,
                                                              18.048764119896610225,
                                                              18.05204705830509937,
                                                              18.055326215810708845,
                                                              18.058601601141383242,
                                                              18.061873222994794475,
                                                              18.065141090038160598,
                                                              18.068405210909475045,
                                                              18.071665594216039352,
                                                              18.074922248536097413,
                                                              18.078175182418121381,
                                                              18.081424404381138515,
                                                              18.084669922914958562,
                                                              18.087911746480227038,
                                                              18.091149883508542473,
                                                              18.094384342402832999,
                                                              18.097615131536880284,
                                                              18.10084225925644219,
                                                              18.104065733878233146,
                                                              18.107285563690883379,
                                                              18.110501756954750618,
                                                              18.113714321902179449,
                                                              18.116923266737305909,
                                                              18.120128599636711186,
                                                              18.123330328748984641,
                                                              18.126528462195601321,
                                                              18.129723008069866808,
                                                              18.132913974438224614,
                                                              18.136101369339865386,
                                                              18.139285200786897434,
                                                              18.142465476764218835,
                                                              18.145642205230174682,
                                                              18.148815394116070365,
                                                              18.15198505132675777,
                                                              18.155151184740685011,
                                                              18.158313802209818277,
                                                              18.161472911559862098,
                                                              18.164628520590550664,
                                                              18.16778063707504387,
                                                              18.170929268761465636,
                                                              18.17407442337134782,
                                                              18.177216108600919853,
                                                              18.180354332120867156,
                                                              18.183489101576274294,
                                                              18.186620424586880773,
                                                              18.189748308747581973,
                                                              18.192872761627686629,
                                                              18.19599379077147816,
                                                              18.199111403698633893,
                                                              18.202225607903937288,
                                                              18.205336410857679397,
                                                              18.208443820005285829,
                                                              18.211547842767703997,
                                                              18.214648486541943129,
                                                              18.217745758700090164,
                                                              18.220839666590617156,
                                                              18.223930217538207188,
                                                              18.227017418842539342,
                                                              18.23010127778079692,
                                                              18.23318180160549673,
                                                              18.236258997545750304,
                                                              18.239332872807583641,
                                                              18.242403434572771914,
                                                              18.245470690000708203,
                                                              18.248534646226922007,
                                                              18.251595310364102431,
                                                              18.254652689501750018,
                                                              18.257706790706624389,
                                                              18.260757621022719377,
                                                              18.263805187471096048,
                                                              18.266849497050291262,
                                                              18.269890556736381626,
                                                              18.272928373482780984,
                                                              18.275962954220982937,
                                                              18.278994305859999514,
                                                              18.282022435286766182,
                                                              18.28504734936614895,
                                                              18.28806905494078805,
                                                              18.291087558831872428,
                                                              18.294102867838830662,
                                                              18.297114988739124897,
                                                              18.300123928288716257,
                                                              18.30312969322262262,
                                                              18.306132290253390948,
                                                              18.30913172607304773,
                                                              18.312128007352608705,
                                                              18.315121140740988182,
                                                              18.318111132867077373,
                                                              18.321097990338280681,
                                                              18.324081719741027285,
                                                              18.327062327641758799,
                                                              18.330039820585078303,
                                                              18.333014205095942373,
                                                              18.335985487678517103,
                                                              18.338953674816348638,
                                                              18.341918772972586993,
                                                              18.344880788590554488,
                                                              18.347839728093219946,
                                                              18.35079559788319159,
                                                              18.353748404343495082,
                                                              18.356698153836823906,
                                                              18.359644852706377804,
                                                              18.36258850727550751,
                                                              18.365529123847672111,
                                                              18.368466708707362756,
                                                              18.371401268118660255,
                                                              18.374332808326904853,
                                                              18.377261335557800948,
                                                              18.380186856017850516,
                                                              18.383109375894512993,
                                                              18.386028901355754073,
                                                              18.388945438550987177,
                                                              18.391858993610199491,
                                                              18.394769572645088829,
                                                              18.397677181747976505,
                                                              18.400581826992706169,
                                                              18.403483514434544333,
                                                              18.406382250109903254,
                                                              18.409278040037300173,
                                                              18.412170890216181363,
                                                              18.415060806628041234,
                                                              18.417947795235875219,
                                                              18.420831861984659383,
                                                              18.423713012801581357,
                                                              18.426591253594740039,
                                                              18.429466590255497493,
                                                              18.432339028656336666,
                                                              18.435208574652552471,
                                                              18.438075234081573228,
                                                              18.440939012762864735,
                                                              18.443799916498530678,
                                                              18.446657951073060389,
                                                              18.449513122253410558,
                                                              18.452365435789261028,
                                                              18.455214897412911768,
                                                              18.458061512839364582,
                                                              18.460905287766870231,
                                                              18.463746227875390105,
                                                              18.466584338829385104,
                                                              18.469419626275136892,
                                                              18.472252095842428332,
                                                              18.475081753144252161,
                                                              18.477908603777017049,
                                                              18.480732653319968506,
                                                              18.483553907336020217,
                                                              18.486372371370983103}};


} /* namespace gum */
