/***************************************************************************
 *  aGrUM modified frames and atg files for cocoR
 *   Copyright (c) 2005-2024 by Pierre-Henri WUILLEMIN(@LIP6) and Christophe GONZALES(@AMU)
 *   info_at_agrum_dot_org
***************************************************************************/
/*----------------------------------------------------------------------
Compiler Generator Coco/R,
Copyright (c) 1990, 2004 Hanspeter Moessenboeck, University of Linz
extended by M. Loeberbauer & A. Woess, Univ. of Linz
ported to C++ by Csaba Balazs, University of Szeged
with improvements by Pat Terry, Rhodes University

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

As an exception, it is allowed to write an extension of Coco/R that is
used as a plugin in non-free software.

If not otherwise stated, any source code generated by Coco/R (other than
Coco/R itself) does not fall under the GNU General Public License.
-----------------------------------------------------------------------*/


#include <iostream>
#include <wchar.h>

#include "Parser.h"
#include "Scanner.h"


namespace gum {
namespace BIF {


void Parser::SynErr( int n ) {
  if ( errDist >= minErrDist ) SynErr( scanner->filename(),la->line, la->col, n );

  errDist = 0;
}


const ErrorsContainer& Parser::errors( void ) const {
  return errors__;
}
ErrorsContainer& Parser::errors( void ) {
  return errors__;
}

void Parser::Get() {
  for ( ;; ) {
    t = la;
    la = scanner->Scan();

    if ( la->kind <= maxT ) { ++errDist; break; }

    

    if ( dummyToken != t ) {
      dummyToken->kind = t->kind;
      dummyToken->pos = t->pos;
      dummyToken->col = t->col;
      dummyToken->line = t->line;
      dummyToken->next = nullptr;
      coco_string_delete( dummyToken->val );
      dummyToken->val = coco_string_create( t->val );
      t = dummyToken;
    }

    la = t;
  }
}

void Parser::Expect( int n ) {
  if ( la->kind==n ) Get(); else { SynErr( n ); }
}

void Parser::ExpectWeak( int n, int follow ) {
  if ( la->kind == n ) Get();
  else {
    SynErr( n );

    while ( !StartOf( follow ) ) Get();
  }
}

bool Parser::WeakSeparator( int n, int syFol, int repFol ) {
  if ( la->kind == n ) {Get(); return true;}
  else if ( StartOf( repFol ) ) {return false;}
  else {
    SynErr( n );

    while ( !( StartOf( syFol ) || StartOf( repFol ) || StartOf( 0 ) ) ) {
      Get();
    }

    return StartOf( syFol );
  }
}

void Parser::BIF() {
		NETWORK();
		while (la->kind == 9 /* "variable" */ || la->kind == 16 /* "probability" */) {
			if (la->kind == 9 /* "variable" */) {
				while (!(la->kind == _EOF || la->kind == 9 /* "variable" */)) {SynErr(26); Get();}
				VARIABLE();
			} else {
				while (!(la->kind == _EOF || la->kind == 16 /* "probability" */)) {SynErr(27); Get();}
				PROBA();
			}
		}
}

void Parser::NETWORK() {
		std::string name_of_network;
		factory().startNetworkDeclaration();
		
		Expect(6 /* "network" */);
		if (la->kind == _ident) {
			IDENT(name_of_network);
		} else if (la->kind == _string) {
			STRING(name_of_network);
		} else SynErr(28);
		factory().addNetworkProperty("name", name_of_network); 
		Expect(7 /* "{" */);
		while (la->kind == 23 /* "property" */) {
			PROPERTY();
		}
		Expect(8 /* "}" */);
		factory().endNetworkDeclaration(); 
}

void Parser::VARIABLE() {
		std::string name_of_var;
		std::string name_of_type;
		Size nbrMod;
		factory().startVariableDeclaration();
		
		Expect(9 /* "variable" */);
		IDENT(name_of_var);
		TRY( factory().variableName(name_of_var)); 
		Expect(7 /* "{" */);
		while (la->kind == 23 /* "property" */) {
			PROPERTY();
		}
		LABELIZE_VAR(nbrMod);
		TRY(factory().endVariableDeclaration());
		Size nbr=0;
		TRY(nbr=factory().varInBN(factory().variableId(name_of_var)).domainSize());
		if (nbrMod<nbr) SemErr("Too much modalities for variable "+name_of_var);
		if (nbrMod>nbr) SemErr("Too many modalities for variable "+name_of_var);
		
		while (la->kind == 23 /* "property" */) {
			PROPERTY();
		}
		Expect(8 /* "}" */);
}

void Parser::PROBA() {
		Expect(16 /* "probability" */);
		std::vector<std::string> var_seq;
		std::string var;
		std::vector<float> proba;
		std::vector<std::string> parents;
		bool error_on_variable=false;
		
		Expect(17 /* "(" */);
		IDENT(var);
		try { factory().variableId(var); } catch (gum::Exception& e) { error_on_variable=true;SemErr(e.errorType());}
		TRY(factory().startParentsDeclaration(var));
		
		if (la->kind == 18 /* "|" */) {
			Get();
			LISTE_PARENTS(parents);
		}
		Expect(19 /* ")" */);
		for (Size i=Size(parents.size());i>=Size(1);--i){
		 TRY(factory().variableId(parents[i-1]));
		 TRY( factory().addParent(parents[i-1]));
		}
		var_seq.push_back(var);
		for (Size i=0;i<Size(parents.size());++i){
		 var_seq.push_back(parents[i]);
		}
		TRY(factory().endParentsDeclaration()); 
		Expect(7 /* "{" */);
		while (la->kind == 23 /* "property" */) {
			PROPERTY();
		}
		if (la->kind == 20 /* "table" */) {
			RAW_PROBA(proba);
			if (! error_on_variable) {
			 TRY(factory().startRawProbabilityDeclaration(var));
			 gum::Size s=(gum::Size)0;
			 TRY(s=factory().cptDomainSize(factory().variableId(var)));
			 if (proba.size()<s) Warning("Not enough data for cpt of node "+var);
			 if (proba.size()>s) Warning("Too many data for cpt of node "+var);
			 TRY(factory().rawConditionalTable(var_seq,proba));
			 TRY(factory().endRawProbabilityDeclaration());
			}
			
			Expect(14 /* ";" */);
		} else if (la->kind == 17 /* "(" */ || la->kind == 21 /* "default" */) {
			FACTORIZED_PROBA(var,parents);
		} else SynErr(29);
		Expect(8 /* "}" */);
}

void Parser::IDENT(std::string& name) {
		Expect(_ident);
		name=narrow(t->val);  
}

void Parser::STRING(std::string& str) {
		Expect(_string);
		str=narrow(t->val); 
}

void Parser::PROPERTY() {
		std::string name_of_prop,content; float val; 
		Expect(23 /* "property" */);
		IDENT(name_of_prop);
		if (la->kind == 24 /* "=" */) {
			Get();
			STRING(content);
			Expect(14 /* ";" */);
		} else if (la->kind == _largestring) {
			Get();
		} else if (la->kind == _ident) {
			IDENT(content);
			Expect(14 /* ";" */);
		} else if (la->kind == _integer || la->kind == _number) {
			FLOAT(val);
			Expect(14 /* ";" */);
		} else SynErr(30);
		Warning("Properties are not supported yet"); 
}

void Parser::LABELIZE_VAR(gum::Size& nbrMod) {
		Expect(10 /* "type" */);
		Expect(11 /* "discrete" */);
		Expect(12 /* "[" */);
		NBR(nbrMod);
		Expect(13 /* "]" */);
		if (nbrMod<=1) SemErr("Not enough modalities for a discrete variable"); 
		Expect(7 /* "{" */);
		MODALITY_LIST();
		Expect(8 /* "}" */);
		Expect(14 /* ";" */);
}

void Parser::NBR(gum::Size& val) {
		Expect(_integer);
		val=coco_atoi(t->val); 
}

void Parser::MODALITY_LIST() {
		std::string label; 
		IDENT_OR_INTEGER(label);
		TRY(factory().addModality(label)); 
		if (la->kind == 15 /* "," */) {
			ExpectWeak(15 /* "," */, 1);
			MODALITY_LIST();
		}
}

void Parser::IDENT_OR_INTEGER(std::string& name) {
		if (la->kind == _ident) {
			IDENT(name);
		} else if (la->kind == _integer) {
			Get();
			name=narrow(t->val);  
		} else SynErr(31);
}

void Parser::LISTE_PARENTS(std::vector<std::string>& parents ) {
		std::string parent; 
		IDENT(parent);
		parents.push_back(parent);
		
		while (la->kind == 15 /* "," */) {
			Get();
			IDENT(parent);
			parents.push_back(parent);
			
		}
}

void Parser::RAW_PROBA(std::vector<float>& v ) {
		Expect(20 /* "table" */);
		v.clear(); 
		LISTE_FLOAT(v);
}

void Parser::FACTORIZED_PROBA(std::string& var,const std::vector<std::string>& parents ) {
		TRY(factory().startFactorizedProbabilityDeclaration(var));
		
		ASSIGNATION(var,parents,true);
		while (la->kind == 17 /* "(" */ || la->kind == 21 /* "default" */) {
			ASSIGNATION(var,parents,false);
		}
		TRY(factory().endFactorizedProbabilityDeclaration()); 
}

void Parser::LISTE_FLOAT(std::vector<float>& v ) {
		float value=0.0; 
		FLOAT(value);
		v.push_back(value); 
		if (StartOf(2)) {
			if (la->kind == 15 /* "," */ || la->kind == 18 /* "|" */) {
				if (la->kind == 15 /* "," */) {
					Get();
				} else {
					Get();
				}
			}
			LISTE_FLOAT(v);
		}
}

void Parser::ASSIGNATION(const std::string& var,const std::vector<std::string>& parents,bool is_first ) {
		std::vector<float> v; std::vector<std::string> labels;
		
		if (la->kind == 17 /* "(" */) {
			Get();
			TRY(factory().startFactorizedEntry()); 
			LISTE_LABELS(parents,labels,0);
			Expect(19 /* ")" */);
			LISTE_FLOAT(v);
			checkSizeOfProbabilityAssignation__(v,var);
			TRY(factory().setVariableValuesUnchecked(v));
			TRY(factory().endFactorizedEntry());
			
		} else if (la->kind == 21 /* "default" */) {
			Get();
			if (! is_first) SemErr("'default' assignation has to be the first."); 
			LISTE_FLOAT(v);
			TRY(factory().startFactorizedEntry());
			checkSizeOfProbabilityAssignation__(v,var);
			TRY(factory().setVariableValuesUnchecked(v));
			TRY(factory().endFactorizedEntry());
			
		} else SynErr(32);
		Expect(14 /* ";" */);
}

void Parser::LISTE_LABELS(const std::vector<std::string>& parents,std::vector<std::string>& labels, Idx num_label ) {
		std::string name_of_label;
		
		if (la->kind == _ident || la->kind == _integer) {
			IDENT_OR_INTEGER(name_of_label);
			labels.push_back(name_of_label);
			if (num_label>=parents.size()) {
			SemErr("Too many labels in this assignation");
			} else {
			TRY(factory().setParentModality(parents[num_label],name_of_label));
			}
			
		} else if (la->kind == 22 /* "*" */) {
			Get();
			labels.push_back("*"); 
		} else SynErr(33);
		if (la->kind == 15 /* "," */) {
			ExpectWeak(15 /* "," */, 3);
			LISTE_LABELS(parents,labels,num_label+1);
		}
}

void Parser::FLOAT(float& val) {
		if (la->kind == _number) {
			Get();
			val=coco_atof(t->val); 
		} else if (la->kind == _integer) {
			Get();
			val=float(coco_atoi(t->val)); 
		} else SynErr(34);
}



// If the user declared a method Init and a mehtod Destroy they should
// be called in the contructur and the destructor respctively.
//
// The following templates are used to recognize if the user declared
// the methods Init and Destroy.

template<typename T>
struct ParserInitExistsRecognizer {
  template<typename U, void ( U::* )() = &U::Init>
  struct ExistsIfInitIsDefinedMarker {};

  struct InitIsMissingType {
    char dummy1;
  };

  struct InitExistsType {
    char dummy1; char dummy2;
  };

  // exists always
  template<typename U>
  static InitIsMissingType is_here( ... );

  // exist only if ExistsIfInitIsDefinedMarker is defined
  template<typename U>
  static InitExistsType is_here( ExistsIfInitIsDefinedMarker<U>* );

  enum { InitExists = ( sizeof( is_here<T>( nullptr ) ) == sizeof( InitExistsType ) ) };
};

template<typename T>
struct ParserDestroyExistsRecognizer {
  template<typename U, void ( U::* )() = &U::Destroy>
  struct ExistsIfDestroyIsDefinedMarker {};

  struct DestroyIsMissingType {
    char dummy1;
  };

  struct DestroyExistsType {
    char dummy1; char dummy2;
  };

  // exists always
  template<typename U>
  static DestroyIsMissingType is_here( ... );

  // exist only if ExistsIfDestroyIsDefinedMarker is defined
  template<typename U>
  static DestroyExistsType is_here( ExistsIfDestroyIsDefinedMarker<U>* );

  enum { DestroyExists = ( sizeof( is_here<T>( nullptr ) ) == sizeof( DestroyExistsType ) ) };
};

// The folloing templates are used to call the Init and Destroy methods if they exist.

// Generic case of the ParserInitCaller, gets used if the Init method is missing
template<typename T, bool = ParserInitExistsRecognizer<T>::InitExists>
struct ParserInitCaller {
  static void CallInit( T* t ) {
    // nothing to do
  }
};

// True case of the ParserInitCaller, gets used if the Init method exists
template<typename T>
struct ParserInitCaller<T, true> {
  static void CallInit( T* t ) {
    t->Init();
  }
};

// Generic case of the ParserDestroyCaller, gets used if the Destroy method is missing
template<typename T, bool = ParserDestroyExistsRecognizer<T>::DestroyExists>
struct ParserDestroyCaller {
  static void CallDestroy( T* t ) {
    // nothing to do
  }
};

// True case of the ParserDestroyCaller, gets used if the Destroy method exists
template<typename T>
struct ParserDestroyCaller<T, true> {
  static void CallDestroy( T* t ) {
    t->Destroy();
  }
};
void Parser::Parse() {
  t = nullptr;
  la = dummyToken = new Token();
  la->val = coco_string_create( L"Dummy Token" );
  Get();
  	BIF();
	Expect(0);
}

Parser::Parser( Scanner* scanner ) {
  	maxT = 25;

  ParserInitCaller<Parser>::CallInit( this );
  dummyToken = nullptr;
  t = la = nullptr;
  minErrDist = 2;
  errDist = minErrDist;
  this->scanner = scanner;
}

bool Parser::StartOf( int s ) {
  const bool T = true;
  const bool x = false;

  	static bool set[4][27] = {
		{T,x,x,x, x,x,x,x, x,T,x,x, x,x,x,x, T,x,x,x, x,x,x,x, x,x,x},
		{T,T,T,x, x,x,x,x, x,T,x,x, x,x,x,x, T,x,x,x, x,x,x,x, x,x,x},
		{x,x,T,T, x,x,x,x, x,x,x,x, x,x,x,T, x,x,T,x, x,x,x,x, x,x,x},
		{T,T,T,x, x,x,x,x, x,T,x,x, x,x,x,x, T,x,x,x, x,x,T,x, x,x,x}
	};



  return set[s][la->kind];
}

Parser::~Parser() {
  ParserDestroyCaller<Parser>::CallDestroy( this );
  delete dummyToken;
}
void Parser::SemErr( const wchar_t* msg ) {
  if ( errDist >= minErrDist ) errors__.Error( scanner->filename(),t->line, t->col, msg );

  errDist = 0;
}

void Parser::Warning( const wchar_t* msg ) {
  errors__.Warning( scanner->filename(),t->line, t->col, msg );
}

void Parser::SynErr( const std::wstring& filename,int line, int col, int n ) {
  wchar_t* s;

  switch ( n ) {
      			case 0: s = coco_string_create(L"EOF expected"); break;
			case 1: s = coco_string_create(L"ident expected"); break;
			case 2: s = coco_string_create(L"integer expected"); break;
			case 3: s = coco_string_create(L"number expected"); break;
			case 4: s = coco_string_create(L"string expected"); break;
			case 5: s = coco_string_create(L"largestring expected"); break;
			case 6: s = coco_string_create(L"\"network\" expected"); break;
			case 7: s = coco_string_create(L"\"{\" expected"); break;
			case 8: s = coco_string_create(L"\"}\" expected"); break;
			case 9: s = coco_string_create(L"\"variable\" expected"); break;
			case 10: s = coco_string_create(L"\"type\" expected"); break;
			case 11: s = coco_string_create(L"\"discrete\" expected"); break;
			case 12: s = coco_string_create(L"\"[\" expected"); break;
			case 13: s = coco_string_create(L"\"]\" expected"); break;
			case 14: s = coco_string_create(L"\";\" expected"); break;
			case 15: s = coco_string_create(L"\",\" expected"); break;
			case 16: s = coco_string_create(L"\"probability\" expected"); break;
			case 17: s = coco_string_create(L"\"(\" expected"); break;
			case 18: s = coco_string_create(L"\"|\" expected"); break;
			case 19: s = coco_string_create(L"\")\" expected"); break;
			case 20: s = coco_string_create(L"\"table\" expected"); break;
			case 21: s = coco_string_create(L"\"default\" expected"); break;
			case 22: s = coco_string_create(L"\"*\" expected"); break;
			case 23: s = coco_string_create(L"\"property\" expected"); break;
			case 24: s = coco_string_create(L"\"=\" expected"); break;
			case 25: s = coco_string_create(L"??? expected"); break;
			case 26: s = coco_string_create(L"this symbol not expected in BIF"); break;
			case 27: s = coco_string_create(L"this symbol not expected in BIF"); break;
			case 28: s = coco_string_create(L"invalid NETWORK"); break;
			case 29: s = coco_string_create(L"invalid PROBA"); break;
			case 30: s = coco_string_create(L"invalid PROPERTY"); break;
			case 31: s = coco_string_create(L"invalid IDENT_OR_INTEGER"); break;
			case 32: s = coco_string_create(L"invalid ASSIGNATION"); break;
			case 33: s = coco_string_create(L"invalid LISTE_LABELS"); break;
			case 34: s = coco_string_create(L"invalid FLOAT"); break;


    default: {
      wchar_t format[20];
      coco_swprintf( format, 20, L"error %d", n );
      s = coco_string_create( format );
    }
    break;
  }

  //wprintf(L"-- line %d col %d: %ls\n", line, col, s);
  std::wstring ss=L"Syntax error : "+std::wstring( s );
  errors__.Error( filename,line,col,ss.c_str() );
  coco_string_delete( s );
}

} // namespace
} // namespace



